@props([
    'options' => [],
    'optionValue' => 'id',
    'optionLabel' => 'name',
    'optionSublabel' => null,
    'optionLabelFormat' => null, // New prop for custom label formatting
    'placeholder' => 'Select options',
    'searchPlaceholder' => 'Search...',
    'wire:model' => null
])

@php
    $wireModel = $attributes->get('wire:model');
    $componentId = 'multiselect_' . uniqid();
    $functionName = 'customMultiselect_' . $componentId;
@endphp

<div class="custom-multiselect" 
     x-data="{{ $functionName }}('{{ $componentId }}', '{{ $wireModel }}', {{ json_encode($options) }}, '{{ $optionValue }}', '{{ $optionLabel }}', '{{ $optionSublabel }}', '{{ $optionLabelFormat }}', '{{ $placeholder }}')"
     x-init="init()"
     wire:ignore.self>
    
    <div class="multiselect-trigger" @click="toggleDropdown()">
        <span class="selected-text" x-text="getSelectedText()">{{ $placeholder }}</span>
        <svg class="arrow-icon" :class="{ 'rotated': isOpen }" width="12" height="12" viewBox="0 0 12 12">
            <path d="M2 4l4 4 4-4" stroke="currentColor" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round"/>
        </svg>
    </div>
    
    <div class="multiselect-dropdown" x-show="isOpen" x-transition @click.away="isOpen = false">
        <div class="multiselect-header">
            <div class="search-container">
                <input type="text" 
                       x-model="searchQuery" 
                       @input="filterOptions()"
                       :placeholder="'{{ $searchPlaceholder }}'"
                       class="search-input"
                       @click.stop>
            </div>
            <label class="select-all-label">
                <input type="checkbox" @change="toggleSelectAll($event)" :checked="isAllSelected()">
                <span>Select All</span>
            </label>
        </div>
        
        <div class="multiselect-options">
            <template x-for="option in filteredOptions" :key="option[optionValue]">
                <label class="multiselect-option">
                    <input type="checkbox" 
                           :value="option[optionValue]" 
                           @change="toggleOption($event, option[optionValue])"
                           :checked="selectedValues.includes(option[optionValue])">
                    <span class="option-text">
                        <span x-text="formatOptionLabel(option)"></span>
                    </span>
                </label>
            </template>
            
            <div x-show="filteredOptions.length === 0" class="no-results">
                No results found
            </div>
        </div>
    </div>
    
    <!-- Hidden input for Livewire binding -->
    <input type="hidden" {{ $attributes->merge(['class' => 'hidden']) }} x-ref="hiddenInput">
</div>

<style>
.custom-multiselect {
    position: relative;
    width: 100%;
}

.multiselect-trigger {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 8px 12px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    background: white;
    cursor: pointer;
    transition: all 0.2s ease;
    min-height: 38px;
}

.multiselect-trigger:hover {
    border-color: #9ca3af;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
}

.multiselect-trigger:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.selected-text {
    color: #374151;
    font-size: 14px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    flex: 1;
}

.arrow-icon {
    color: #6b7280;
    transition: transform 0.2s ease;
    flex-shrink: 0;
    margin-left: 8px;
}

.arrow-icon.rotated {
    transform: rotate(180deg);
}

.multiselect-dropdown {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    background: white;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
    z-index: 1000;
    max-height: 400px;
    overflow: hidden;
    margin-top: 2px;
}

.multiselect-header {
    padding: 12px 16px;
    border-bottom: 1px solid #e5e7eb;
    background: #f9fafb;
}

.search-container {
    margin-bottom: 12px;
}

.search-input {
    width: 100%;
    padding: 8px 12px;
    border: 1px solid #d1d5db;
    border-radius: 4px;
    font-size: 14px;
    transition: border-color 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 2px rgba(59, 130, 246, 0.1);
}

.select-all-label {
    display: flex;
    align-items: center;
    gap: 8px;
    cursor: pointer;
    font-weight: 500;
    color: #374151;
}

.select-all-label input[type="checkbox"] {
    width: 16px;
    height: 16px;
    accent-color: #3b82f6;
}

.multiselect-options {
    max-height: 300px;
    overflow-y: auto;
}

.multiselect-option {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    cursor: pointer;
    transition: background-color 0.15s ease;
    border-bottom: 1px solid #f3f4f6;
}

.multiselect-option:hover {
    background-color: #f3f4f6;
}

.multiselect-option:last-child {
    border-bottom: none;
}

.multiselect-option input[type="checkbox"] {
    width: 16px;
    height: 16px;
    accent-color: #3b82f6;
    flex-shrink: 0;
}

.option-text {
    color: #374151;
    font-size: 14px;
    line-height: 1.4;
    flex: 1;
}

.option-sublabel {
    color: #6b7280;
    font-size: 13px;
}

.no-results {
    padding: 16px;
    text-align: center;
    color: #6b7280;
    font-style: italic;
}

/* Custom scrollbar for the dropdown */
.multiselect-dropdown::-webkit-scrollbar {
    width: 6px;
}

.multiselect-dropdown::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 3px;
}

.multiselect-dropdown::-webkit-scrollbar-thumb {
    background: #c1c1c1;
    border-radius: 3px;
}

.multiselect-dropdown::-webkit-scrollbar-thumb:hover {
    background: #a8a8a8;
}

/* Animation for dropdown */
.multiselect-dropdown[x-cloak] {
    display: none;
}
</style>

<script>
function {{ $functionName }}(componentId, wireModel, options, optionValue, optionLabel, optionSublabel, optionLabelFormat, placeholder) {
    return {
        componentId,
        wireModel,
        options,
        optionValue,
        optionLabel,
        optionSublabel,
        optionLabelFormat,
        placeholder,
        isOpen: false,
        selectedValues: [],
        searchQuery: '',
        filteredOptions: [],
        
        init() {
            // console.log('Multiselect component initialized');
            // console.log('Component ID:', this.componentId);
            // console.log('Wire model:', this.wireModel);
            // console.log('Options:', this.options);
            // console.log('Option value field:', this.optionValue);
            // console.log('Option label field:', this.optionLabel);
            // console.log('Options count:', this.options.length);
            
            this.filteredOptions = [...this.options];
            
            // Initialize with any pre-selected values from Livewire
            // Check if the hidden input has a value
            const hiddenValue = this.$refs.hiddenInput.value;
            if (hiddenValue && hiddenValue.trim() !== '') {
                this.selectedValues = hiddenValue.split(',').filter(v => v.trim() !== '');
            }
            
            // Ensure selectedValues is always an array
            if (!Array.isArray(this.selectedValues)) {
                this.selectedValues = [];
            }
            
            // Listen for Livewire events to refresh the component
            document.addEventListener('livewire:load', () => {
                this.refreshOptions();
            });
            
            // Listen for specific Livewire update events
            document.addEventListener('livewire:update', () => {
                setTimeout(() => {
                    this.refreshOptions();
                }, 100);
            });
            
            // Listen for custom multiselect refresh event
            document.addEventListener('multiselect-refresh', () => {
                setTimeout(() => {
                    this.refreshOptions();
                }, 100);
            });
            
            // Watch for changes in the options from Livewire
            this.$watch('options', (newOptions) => {
                // console.log('Options changed:', newOptions);
                if (newOptions && Array.isArray(newOptions)) {
                    this.filteredOptions = [...newOptions];
                    // Re-filter if there's a search query
                    if (this.searchQuery.trim()) {
                        this.filterOptions();
                    }
                }
            });
            
            // Watch for changes in the wire:model value
            if (this.wireModel) {
                this.$watch('$wire.' + this.wireModel, (newValue) => {
                    // console.log('Wire model value changed:', newValue);
                    if (newValue && typeof newValue === 'string') {
                        this.selectedValues = newValue.split(',').filter(v => v.trim() !== '');
                    } else if (Array.isArray(newValue)) {
                        this.selectedValues = [...newValue];
                    } else {
                        this.selectedValues = [];
                    }
                });
            }
            
            // console.log('Initialization complete');
        },
        
        refreshOptions() {
            // Force refresh of options and filtered options
            // console.log('Refreshing options:', this.options);
            this.filteredOptions = [...this.options];
            if (this.searchQuery.trim()) {
                this.filterOptions();
            }
            
            // Debug: Log the filtered options
            // console.log('Filtered options:', this.filteredOptions);
            // console.log('Options count:', this.options.length);
            // console.log('Filtered count:', this.filteredOptions.length);
        },
        
        toggleDropdown() {
            this.isOpen = !this.isOpen;
            if (this.isOpen) {
                this.searchQuery = '';
                this.filterOptions();
            }
        },
        
        filterOptions() {
            if (!this.searchQuery.trim()) {
                this.filteredOptions = [...this.options];
            } else {
                const query = this.searchQuery.toLowerCase();
                this.filteredOptions = this.options.filter(option => {
                    // Search in the formatted label
                    const label = this.formatOptionLabel(option).toLowerCase();
                    return label.includes(query);
                });
            }
        },
        
        formatOptionLabel(option) {
            // Debug: Log the option being processed
            // console.log('Formatting option:', option);
            // console.log('Option value:', option[this.optionValue]);
            // console.log('Option label:', option[this.optionLabel]);
            
            // If custom format is provided, use it
            if (this.optionLabelFormat) {
                return this.optionLabelFormat
                    .replace(/\{(\w+)\}/g, (match, field) => {
                        return option[field] || '';
                    })
                    .replace(/\s+/g, ' ')
                    .trim();
            }
            
            // Default formatting with main label and sublabel
            let label = option[this.optionLabel] || '';
            
            if (this.optionSublabel && option[this.optionSublabel]) {
                label += ' - ' + option[this.optionSublabel];
            }
            
            // console.log('Final formatted label:', label);
            return label;
        },
        
        toggleOption(event, value) {
            const checkbox = event.target;
            if (checkbox.checked) {
                if (!this.selectedValues.includes(value)) {
                    this.selectedValues.push(value);
                }
            } else {
                this.selectedValues = this.selectedValues.filter(v => v !== value);
            }
            this.updateLivewire();
        },
        
        toggleSelectAll(event) {
            const checkbox = event.target;
            if (checkbox.checked) {
                // Select all visible options
                this.selectedValues = [...new Set([...this.selectedValues, ...this.filteredOptions.map(opt => opt[this.optionValue])])];
            } else {
                // Deselect all visible options
                this.selectedValues = this.selectedValues.filter(value => 
                    !this.filteredOptions.some(opt => opt[this.optionValue] === value)
                );
            }
            this.updateLivewire();
        },
        
        isAllSelected() {
            if (this.filteredOptions.length === 0) return false;
            return this.filteredOptions.every(option => 
                this.selectedValues.includes(option[this.optionValue])
            );
        },
        
        getSelectedText() {
            if (this.selectedValues.length === 0) {
                return this.placeholder;
            } else if (this.selectedValues.length === this.options.length) {
                return 'All Selected';
            } else {
                return `${this.selectedValues.length} Selected`;
            }
        },
        
        updateLivewire() {
            // Update the hidden input for form submission
            this.$refs.hiddenInput.value = this.selectedValues.join(',');
            
            // Trigger Livewire update by dispatching an input event
            this.$refs.hiddenInput.dispatchEvent(new Event('input', { bubbles: true }));
            
            // Also trigger a change event to ensure Livewire picks it up
            this.$refs.hiddenInput.dispatchEvent(new Event('change', { bubbles: true }));
        }
    }
}
</script>
