// Tom Select integration with Livewire
import TomSelect from 'tom-select';

// Store Tom Select instances
const tomSelectInstances = new Map();

// Initialize Tom Select with Livewire support
function initializeTomSelect(selectElement, options = {}) {
    // Destroy existing instance if it exists
    if (tomSelectInstances.has(selectElement)) {
        tomSelectInstances.get(selectElement).destroy();
        tomSelectInstances.delete(selectElement);
    }

    // Default options
    const defaultOptions = {
        plugins: ['dropdown_input'],
        create: false,
        sortField: {
            field: "text",
            direction: "asc"
        },
        onChange: function(value) {
            // Trigger Livewire change event
            const event = new CustomEvent('change', {
                detail: { value: value },
                bubbles: true
            });
            selectElement.dispatchEvent(event);
        }
    };

    // Merge options
    const finalOptions = { ...defaultOptions, ...options };
    
    // Create new instance
    const instance = new TomSelect(selectElement, finalOptions);
    tomSelectInstances.set(selectElement, instance);
    
    return instance;
}

// Initialize all Tom Select elements
function initializeAllTomSelects() {
    // Basic Tom Select
    const basicSelects = document.querySelectorAll('.tom-select');
    basicSelects.forEach(select => {
        initializeTomSelect(select);
    });

    // Tom Select with search
    const searchSelects = document.querySelectorAll('.tom-select-search');
    searchSelects.forEach(select => {
        initializeTomSelect(select, {
            searchField: ['text', 'value']
        });
    });

    // Tom Select with multiple selection
    const multiSelects = document.querySelectorAll('.tom-select-multi');
    multiSelects.forEach(select => {
        initializeTomSelect(select, {
            plugins: ['dropdown_input', 'remove_button'],
            searchField: ['text', 'value']
        });
    });

    // Tom Select with remote data loading
    const remoteSelects = document.querySelectorAll('.tom-select-remote');
    remoteSelects.forEach(select => {
        const url = select.dataset.url;
        if (url) {
            initializeTomSelect(select, {
                searchField: ['text', 'value'],
                load: function(query, callback) {
                    const searchUrl = `${url}?search=${encodeURIComponent(query)}`;
                    fetch(searchUrl)
                        .then(response => response.json())
                        .then(data => {
                            callback(data);
                        })
                        .catch(() => {
                            callback();
                        });
                }
            });
        }
    });
}

// Handle Livewire updates
document.addEventListener('livewire:init', () => {
    // Initialize Tom Select after Livewire loads
    initializeAllTomSelects();
    
    // Re-initialize after Livewire updates
    document.addEventListener('livewire:updated', () => {
        console.log('livewire:updated ... ');
        // Small delay to ensure DOM is ready
        setTimeout(() => {
            initializeAllTomSelects();
        }, 10);
    });
});

// Initialize on DOM ready
document.addEventListener('DOMContentLoaded', function() {
    initializeAllTomSelects();
});

// Export for use in other modules
export { TomSelect, initializeTomSelect, initializeAllTomSelects }; 