document.addEventListener('DOMContentLoaded', function() {
    // Initialize custom select components
    initializeCustomSelects();
    
    // Listen for Livewire updates
    document.addEventListener('livewire:load', function() {
        Livewire.hook('message.processed', (message, component) => {
            initializeCustomSelects();
        });
    });
});

function initializeCustomSelects() {
    const customSelects = document.querySelectorAll('.custom-select-container');
    
    customSelects.forEach(container => {
        const wrapper = container.querySelector('.custom-select-wrapper');
        const input = container.querySelector('.custom-select-input');
        const dropdown = container.querySelector('.custom-select-dropdown');
        const options = container.querySelectorAll('.custom-select-option');
        const hiddenSelect = container.querySelector('.custom-select-hidden');
        
        if (!wrapper || !input || !dropdown || !hiddenSelect) return;
        
        let selectedValues = [];
        
        // Initialize selected values from hidden select
        Array.from(hiddenSelect.options).forEach(option => {
            if (option.selected) {
                selectedValues.push(option.value);
            }
        });
        
        // Update display
        updateDisplay();
        
        // Input focus handler
        input.addEventListener('focus', function() {
            showDropdown();
        });
        
        // Input blur handler
        input.addEventListener('blur', function(e) {
            // Delay to allow for option clicks
            setTimeout(() => {
                hideDropdown();
            }, 200);
        });
        
        // Input keydown handler
        input.addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                const filteredOptions = getFilteredOptions();
                if (filteredOptions.length > 0) {
                    selectOption(filteredOptions[0]);
                }
            } else if (e.key === 'Backspace' && input.value === '' && selectedValues.length > 0) {
                e.preventDefault();
                removeLastTag();
            }
        });
        
        // Input input handler for filtering
        input.addEventListener('input', function() {
            filterOptions();
        });
        
        // Option click handlers
        options.forEach(option => {
            option.addEventListener('click', function() {
                const value = this.dataset.value;
                if (selectedValues.includes(value)) {
                    removeValue(value);
                } else {
                    addValue(value);
                }
                input.focus();
            });
        });
        
        // Click outside to close dropdown
        document.addEventListener('click', function(e) {
            if (!container.contains(e.target)) {
                hideDropdown();
            }
        });
        
        function showDropdown() {
            dropdown.classList.add('show');
            filterOptions();
        }
        
        function hideDropdown() {
            dropdown.classList.remove('show');
        }
        
        function filterOptions() {
            const searchTerm = input.value.toLowerCase();
            let hasVisibleOptions = false;
            
            options.forEach(option => {
                const text = option.textContent.toLowerCase();
                const value = option.dataset.value;
                const isSelected = selectedValues.includes(value);
                
                if (text.includes(searchTerm) && !isSelected) {
                    option.style.display = 'block';
                    hasVisibleOptions = true;
                } else {
                    option.style.display = 'none';
                }
            });
            
            if (!hasVisibleOptions) {
                hideDropdown();
            } else {
                showDropdown();
            }
        }
        
        function getFilteredOptions() {
            return Array.from(options).filter(option => 
                option.style.display !== 'none' && 
                !selectedValues.includes(option.dataset.value)
            );
        }
        
        function selectOption(optionElement) {
            const value = optionElement.dataset.value;
            addValue(value);
        }
        
        function addValue(value) {
            if (!selectedValues.includes(value)) {
                selectedValues.push(value);
                updateDisplay();
                updateHiddenSelect();
                triggerLivewireUpdate();
            }
        }
        
        function removeValue(value) {
            selectedValues = selectedValues.filter(v => v !== value);
            updateDisplay();
            updateHiddenSelect();
            triggerLivewireUpdate();
        }
        
        function removeLastTag() {
            if (selectedValues.length > 0) {
                selectedValues.pop();
                updateDisplay();
                updateHiddenSelect();
                triggerLivewireUpdate();
            }
        }
        
        function updateDisplay() {
            // Clear existing tags
            const existingTags = wrapper.querySelectorAll('.email-tag');
            existingTags.forEach(tag => tag.remove());
            
            // Add tags for selected values
            selectedValues.forEach(value => {
                const option = Array.from(options).find(opt => opt.dataset.value === value);
                if (option) {
                    const tag = createTag(option.textContent, value);
                    wrapper.insertBefore(tag, input);
                }
            });
            
            // Update input placeholder
            const placeholderValue = input.getAttribute('placeholder');
            if (selectedValues.length === 0) {
                input.placeholder = placeholderValue;
            } else {
                input.placeholder = placeholderValue;
            }
        }
        
        function createTag(text, value) {
            const tag = document.createElement('div');
            tag.className = 'email-tag';
            tag.innerHTML = `
                <span>${text}</span>
                <button type="button" class="remove-btn" data-value="${value}">×</button>
            `;
            
            // Add remove handler
            const removeBtn = tag.querySelector('.remove-btn');
            removeBtn.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                removeValue(value);
                input.focus();
            });
            
            return tag;
        }
        
        function updateHiddenSelect() {
            // Clear existing selections
            Array.from(hiddenSelect.options).forEach(option => {
                option.selected = false;
            });
            
            // Set new selections
            selectedValues.forEach(value => {
                const option = Array.from(hiddenSelect.options).find(opt => opt.value === value);
                if (option) {
                    option.selected = true;
                }
            });
        }
        
        function triggerLivewireUpdate() {
            // Trigger Livewire model update
            const event = new Event('change', { bubbles: true });
            hiddenSelect.dispatchEvent(event);
        }
    });
} 