# Migration Notes for Shared Database

## Important Information

This application shares a database with another Laravel application. This means that some tables may already exist in your database.

## If Tables Already Exist

### Option 1: Skip Migrations (Recommended)

If your other Laravel app has already created the `users`, `sessions`, and other tables, you can simply **skip running migrations** in this app.

The application will work with the existing tables as long as they have the standard Laravel structure.

### Option 2: Conditional Migrations

If you need to run migrations but want to skip existing tables, you can modify the migration files to check if tables exist first.

Example for `0001_01_01_000000_create_users_table.php`:

```php
public function up(): void
{
    // Only create tables if they don't exist
    if (!Schema::hasTable('users')) {
        Schema::create('users', function (Blueprint $table) {
            // ... table definition
        });
    }
    
    if (!Schema::hasTable('password_reset_tokens')) {
        Schema::create('password_reset_tokens', function (Blueprint $table) {
            // ... table definition
        });
    }
    
    if (!Schema::hasTable('sessions')) {
        Schema::create('sessions', function (Blueprint $table) {
            // ... table definition
        });
    }
}
```

### Option 3: Mark Migrations as Run

If you want to use Laravel's migration tracking but tables already exist:

```bash
# This will mark all migrations as run without actually running them
php artisan migrate --pretend
```

Or manually insert records into the `migrations` table in your database.

## Table Structure Requirements

For shared authentication to work, ensure these tables exist with the standard Laravel structure:

### `users` table
- id (bigint, primary key)
- name (varchar)
- email (varchar, unique)
- email_verified_at (timestamp, nullable)
- password (varchar)
- remember_token (varchar, nullable)
- created_at (timestamp)
- updated_at (timestamp)

### `sessions` table
- id (varchar, primary key)
- user_id (bigint, nullable, indexed)
- ip_address (varchar, nullable)
- user_agent (text, nullable)
- payload (longtext)
- last_activity (integer, indexed)

### Optional: `password_reset_tokens` table
- email (varchar, primary key)
- token (varchar)
- created_at (timestamp, nullable)

## Verifying Your Setup

Check if tables exist in your shared database:

```sql
SHOW TABLES;

-- Check users table structure
DESCRIBE users;

-- Check sessions table structure
DESCRIBE sessions;
```

## If Your Tables Have Different Structure

If your existing `users` or `sessions` tables have a different structure, you may need to:

1. **Adjust the User Model** (`app/Models/User.php`) to match your table structure
2. **Add custom column mappings** in your model
3. **Create database views** that map your structure to Laravel's expected structure

Example of custom attribute mapping in User model:

```php
class User extends Authenticatable
{
    // If your table has different column names
    protected $table = 'your_users_table_name';
    
    // Map Laravel's expected attributes to your column names
    const CREATED_AT = 'creation_date';
    const UPDATED_AT = 'modification_date';
    
    // Additional customizations as needed
}
```

## Running Migrations Safely

If you decide to run migrations:

```bash
# 1. Check what will be run (dry run)
php artisan migrate --pretend

# 2. Run migrations one at a time
php artisan migrate --step

# 3. Or run all migrations
php artisan migrate
```

## Rolling Back

If you accidentally run migrations and need to roll back:

```bash
# Roll back the last batch
php artisan migrate:rollback

# Roll back all migrations
php artisan migrate:reset
```

**Warning:** Rolling back will drop tables, which will affect your other application if they share the same tables!

## Best Practice

The safest approach when working with a shared database:

1. Let the first/main application handle all migrations
2. Don't run migrations in secondary applications
3. Ensure all applications use consistent table structures
4. Document any schema changes across all applications
