<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    // Role constants
    const ROLE_ADMIN = 1;
    const ROLE_OFFICE_ADMIN = 2;
    const ROLE_ADJUSTER = 3;
    const ROLE_APPRAISER = 4;
    const ROLE_SUPERVISOR = 5;


    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
        'phone_number',
        'street',
        'city',
        'state_id',
        'postcode',
        'franchise_id',
        'multiple_franchise',
        'insurance_company_id',
        'username',
        'new_assignment_notification',
        'completed_assignment_notification',
        'supplement_opened_notification',
        'mileage_request_notification',
        'appointment_set_notification',
        'pdf_package_notification',
        'archived',
        'file_first_color',
        'file_second_color',
        'file_third_color',
        'file_fourth_color',
        'current_session_id',
        'appraiser_freemiles',
        'appraiser_feepercentage',
        'franchise_access',
        'mark_as_paid',
        'user_picture',
        'rating'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
    ];

    /**
     * Check if user has a specific role
     */
    public function hasRole($role): bool
    {
        return $this->role === $role;
    }

    /**
     * Check if user is admin
     */
    public function isAdmin(): bool
    {
        return $this->role === self::ROLE_ADMIN;
    }

    /**
     * Check if user is office admin
     */
    public function isOfficeAdmin(): bool
    {
        return $this->role === self::ROLE_OFFICE_ADMIN;
    }

    /**
     * Check if user is adjuster
     */
    public function isAdjuster(): bool
    {
        return $this->role === self::ROLE_ADJUSTER;
    }

    /**
     * Check if user is appraiser
     */
    public function isAppraiser(): bool
    {
        return $this->role === self::ROLE_APPRAISER;
    }

    /**
     * Get role name as string
     */
    public function getRoleName(): string
    {
        return match ($this->role) {
            self::ROLE_ADMIN => 'Admin',
            self::ROLE_OFFICE_ADMIN => 'Office Admin',
            self::ROLE_ADJUSTER => 'Adjuster',
            self::ROLE_APPRAISER => 'Appraiser',
            default => 'Unknown',
        };
    }

    public function insuranceCompany()
    {
        return $this->belongsTo(InsuranceCompany::class);
    }

    /**
     * Get the appraiser fees for this user.
     */
    public function appraiserFees()
    {
        return $this->hasMany(AppraiserFee::class, 'appraiser_id');
    }

    public function franchise()
    {
        return $this->belongsTo(Franchise::class);
    }

    /**
     * Get multiple franchises as an array
     */
    public function getMultipleFranchisesAttribute()
    {
        if (empty($this->multiple_franchise)) {
            return [];
        }
        
        return explode(',', $this->multiple_franchise);
    }

    /**
     * Get all franchises (primary + multiple) as an array
     */
    public function getAllFranchisesAttribute()
    {
        $franchises = [];
        
        if ($this->franchise_id) {
            $franchises[] = $this->franchise_id;
        }
        
        if (!empty($this->multiple_franchise)) {
            $multipleFranchises = explode(',', $this->multiple_franchise);
            $franchises = array_merge($franchises, $multipleFranchises);
        }
        
        return array_unique($franchises);
    }

    /**
     * Get the user's login history.
     */
    public function loginHistory()
    {
        return $this->hasMany(\App\Models\UserLoginHistory::class);
    }

    /**
     * Get the user's recent login history.
     */
    public function recentLoginHistory($days = 30)
    {
        return $this->loginHistory()->recent($days);
    }

    /**
     * Get the user's last login.
     */
    public function lastLogin()
    {
        return $this->loginHistory()
            ->where('action', \App\Models\UserLoginHistory::ACTION_LOGIN)
            ->latest('login_at')
            ->first();
    }

    /**
     * Get the user's last logout.
     */
    public function lastLogout()
    {
        return $this->loginHistory()
            ->whereIn('action', [\App\Models\UserLoginHistory::ACTION_LOGOUT, \App\Models\UserLoginHistory::ACTION_SESSION_TIMEOUT])
            ->latest('logout_at')
            ->first();
    }

    public function state()
    {
        return $this->belongsTo(State::class, 'state_id');
    }

    public function permissions()
    {
        return $this->hasMany(\App\Models\AdminPermission::class, 'admin_id');
    }

    public function hasPermission($permission)
    {
        return $this->permissions->pluck('permission')->contains($permission);
    }

    public function appraiserMiles($vehicle_latitude, $vehicle_longitude)
    {
        $state = $this->state->name ?? '';
        $lan_lon = getLatLongByAddressApi($this->street . ", " . $this->city . ", " . $state . ", " . $this->postcode);
        $lan1 = $lan_lon[0][0] ?? 0;
        $lon1 = $lan_lon[0][1] ?? 0;
        $distanceinmiles = round(distanceCalculation($lan1, $lon1, $vehicle_latitude, $vehicle_longitude), 2);
        return $distanceinmiles ?? 0;
    }
}
