<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Assignment extends Model
{
    use HasFactory;

    protected $table = 'assignments';

    protected $fillable = [
        'file_no',
        'claim_number',
        'policy_number',
        'deductible_amount',
        'insurance_company_id',
        'adjuster_id',
        'date_of_loss',
        'assignment_type',
        'loss_type_id',
        'childlosstype',
        'claimfor',
        'insured_name',
        'business_name',
        'business_first_name',
        'business_last_name',
        'business_email_address',
        'business_phone_number',
        'additional_phone_number',
        'business_contact_name',
        'business_street',
        'business_city',
        'business_state_id',
        'business_postcode',
        'vehicle_location',
        'vehicle_location_name',
        'vehicle_phone_number',
        'vehicle_phone_carrier',
        'vehicle_phone_type',
        'additional_shop_info',
        'vehicle_street',
        'vehicle_city',
        'vehicle_state_id',
        'vehicle_postcode',
        'shop_from_time',
        'shop_to_time',
        'shop_lunch_from',
        'shop_lunch_to',
        'shop_license',
        'shop_tax_id',
        'location_fax',
        'shop_contact_name',
        'shop_email',
        'shop_phone',
        'appointment_date',
        'appraiser_id',
        'webassignment',
        'supplement',
        'web_insurance_company',
        'web_adjuster_first_name',
        'web_adjuster_last_name',
        'web_adjuster_email_address',
        'web_adjuster_phone_number',
        'status_id',
        'assignment_sub_status',
        'office_approve',
        'office_approve_date',
        'appraiser_approve',
        'appraiser_approve_date',
        'auto_reject',
        'phase',
        'phase_date',
        'complete_date',
        'cancel_reason',
        'cancel_date',
        'claim_type_id',
        'unit_type_id',
        'assignment_priority',
        'priority_personname',
        'priority_additionalcharge',
        'instruction',
        'latitude',
        'longitude',
        'created_by',
        'created_date',
        'franchise_id',
        'assignment_notepad',
        'unlock_for',
        'unlock_date',
        'unlock_expire_date',
        'call_type',
        'catastrophe',
        'admin_notes',
        'property_damage',
        'filepath',
        'smartsheet_rowid',
        'smartsheet_fetch_date',
        'sheet_rowid',
        'sheet_fetch_date',
        'vehicle_inspected',
        'phone_number_work',
    ];

    /**
     * Get the insurance company that owns the assignment.
     */
    public function insuranceCompany()
    {
        return $this->belongsTo(InsuranceCompany::class);
    }

    /**
     * Get the adjuster (user) assigned to this assignment.
     */
    public function adjuster()
    {
        return $this->belongsTo(User::class, 'adjuster_id');
    }

    /**
     * Get the appraiser (user) assigned to this assignment.
     */
    public function appraiser()
    {
        return $this->belongsTo(User::class, 'appraiser_id');
    }

    /**
     * Get the loss type for this assignment.
     */
    public function lossType()
    {
        return $this->belongsTo(LossType::class);
    }

    /**
     * Get the claim type for this assignment.
     */
    public function claimType()
    {
        return $this->belongsTo(ClaimType::class);
    }

    /**
     * Get the unit type for this assignment.
     */
    public function unitType()
    {
        return $this->belongsTo(UnitType::class);
    }

    /**
     * Get the franchise that owns this assignment.
     */
    public function franchise()
    {
        return $this->belongsTo(Franchise::class);
    }

    /**
     * Get the user who created this assignment.
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the business state for this assignment.
     */
    public function businessState()
    {
        return $this->belongsTo(State::class, 'business_state_id');
    }

    /**
     * Get the vehicle state for this assignment.
     */
    public function vehicleState()
    {
        return $this->belongsTo(State::class, 'vehicle_state_id');
    }

    /**
     * Get the messages for this assignment.
     */
    public function messages()
    {
        return $this->hasMany(AssignmentMessages::class);
    }

    /**
     * Get the status for this assignment.
     */
    public function status()
    {
        return $this->belongsTo(GlobalStatus::class, 'status_id');
    }

    /**
     * Get the vehicles for this assignment.
     */
    public function vehicles()
    {
        return $this->hasMany(AssignmentVehicle::class);
    }

    /**
     * Get the logs for this assignment.
     */
    public function logs()
    {
        return $this->hasMany(AssignmentLog::class)->orderBy('created_at', 'desc');
    }

    /**
     * Get the invoices for this assignment.
     * Note: Each assignment can only have one invoice.
     */
    public function invoices()
    {
        return $this->hasMany(AssignmentInvoice::class);
    }

    /**
     * Get the single invoice for this assignment.
     * Since each assignment can only have one invoice, this method returns the first (and only) invoice.
     */
    public function invoice()
    {
        return $this->hasOne(AssignmentInvoice::class);
    }

    /**
     * Get the appraisal report data for this assignment.
     */
    public function appraisalReportData()
    {
        return $this->hasOne(AppraisalReportData::class);
    }


    /**
     * Get the phase for this assignment.
     */
    public function phase()
    {
        return $this->belongsTo(Phase::class, 'phase_id');
    }

    /**
     * Get all mileage requests for this assignment.
     */
    public function mileageRequests()
    {
        return $this->hasMany(AssignmentMileage::class, 'assignment_id');
    }

    /**
     * Get all files for this assignment.
     */
    public function files()
    {
        return $this->hasMany(\App\Models\AssignmentFile::class, 'assignment_id');
    }


    /**
     * Get the most recent mileage request model for this assignment.
     */
    public function latestMileageRequest()
    {
        return $this->mileageRequests()
            ->orderByDesc('id')
            ->first();
    }

    /**
     * Get the most recent mileage request status for this assignment.
     */
    public function latestMileageStatus(): ?string
    {
        $latestRequest = $this->latestMileageRequest();
        if ($latestRequest)
        {
            return $latestRequest->request_status;
        }
        else
        {
            return '1';
        }

        // return $latestRequest ? $latestRequest->request_status : null;
    }

}
