<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use App\Models\UnitType;
use Livewire\WithPagination;
use Symfony\Component\HttpFoundation\StreamedResponse;

class TaxReport extends Component
{
    use WithPagination;

    public $from_date;
    public $to_date;
    public $franchise = [];
    public $insurance_company = [];
    public $adjuster_id = [];
    public $appraiser_id = [];
    public $unit_types = [];
    public $unit_type = '';
    public $global_statuses = [];
    public $global_status = '';
    public $isLoading = false;
    public $perPage = 25;
    public $multiple_franchise = [];

    protected $paginationTheme = 'bootstrap';

    public function mount()
    {
        // If user is office admin, then limit the records to the multiple franchise only;
        if (auth()->user()->franchise != 0 && auth()->user()->franchise != '' && auth()->user()->franchise != null)
        {
            $franchises = auth()->user()->multiple_franchise;
            $this->multiple_franchise = $franchises 
                ? array_map('trim', explode(',', $franchises)) 
                : [];
            $this->multiple_franchise[] = auth()->user()->franchise;
            $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
        }

        $this->franchise = [];
        $this->insurance_company = [];
        $this->adjuster_id = [];
        $this->appraiser_id = [];
        $this->unit_type = '';
        $this->global_status = '';
    }

    public function updated($property, $value)
    {
        \Log::info('updated property: '.$property);

        // Set loading to true for any filter change
        $this->isLoading = true;

        $fields = ['from_date', 'to_date', 'franchise', 'insurance_company', 'unit_type', 'global_status'];

        if (in_array($property, $fields)) {
            $this->resetPage();
        }
    }

    public function render()
    {
        try {
            $allAdjusters = DB::select(" SELECT * FROM users join insurance_company on insurance_company.id =users.insurance_company_id AND users.role=4   order by users.name asc ");
            $allAppraisers = DB::select(" SELECT * FROM users WHERE users.role=2   order by users.name asc ");
            if (count($this->multiple_franchise) > 0) {
                $allFranchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->whereIn('id', $this->multiple_franchise)->get();
            } else {
                $allFranchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->get();
            }
            $allInsurances = DB::table('insurance_company')->orderBy('company_name', 'asc')->get();
            $this->unit_types = UnitType::all();
            $this->global_statuses = GlobalStatus::all();

            // Build query using Laravel Query Builder for proper pagination
            $query = DB::table('assignment_invoices')
                ->join('assignments', 'assignments.id', '=', 'assignment_invoices.assignment_id')
                // ->join('tb_invoice_transactions', 'tb_invoice_transactions.invoiceID', '=', 'assignment_invoices.invoiceID')
                ->where('assignment_invoices.invoice_status', 'Paid');

            if (count($this->multiple_franchise) > 0) {
                $query->whereIn('assignments.franchise_id', $this->multiple_franchise);
            }

            // Date filters
            if($this->from_date && $this->to_date)
            {
                $query->whereBetween('assignment_invoices.received_date', [
                    $this->from_date . ' 00:00:00',
                    $this->to_date . ' 23:59:59'
                ]);
            }
            elseif ($this->from_date && !$this->to_date)
            {
                $query->where('assignment_invoices.received_date', '>=', $this->from_date . ' 00:00:00');
            }
            elseif ($this->to_date && !$this->from_date)
            {
                $query->where('assignment_invoices.received_date', '<=', $this->to_date . ' 23:59:59');
            }

            // Franchise filter
            if (!empty($this->franchise)) {
                if (!is_array($this->franchise)) {
                    $query->whereIn('assignments.franchise_id', explode(',', $this->franchise));
                } else {
                    $query->whereIn('assignments.franchise_id', $this->franchise);
                }
            }

            // Insurance Company filter
            if (!empty($this->insurance_company)) {
                $query->whereIn('assignments.insurance_company_id', explode(',', $this->insurance_company));
            }

            // Unit Type filter
            if (!empty($this->unit_type)) {
                $query->whereIn('assignments.unit_type_id', explode(',', $this->unit_type));
            }



            // Select all columns and order by assignment ID
            $query->select([
                'assignment_invoices.*',
                'assignments.*',
                // 'tb_invoice_transactions.paymentDate',
                'assignment_invoices.invoice_status'
            ])
                ->where('assignment_invoices.taxpercentage', '>', 0)

                ->orderBy('assignments.id', 'DESC');

            // Get paginated results
            $data = $query->paginate($this->perPage);

            // Reset loading state
            $this->isLoading = false;

            return view('livewire.tax-report', [
                'franchises' => $allFranchises,
                'insuranceCompanies' => $allInsurances,
                'adjusters' => $allAdjusters,
                'appraisers' => $allAppraisers,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            \Log::error('ReportTax Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            // Return empty data on error - create a proper paginated result
            $emptyQuery = DB::table('assignment_invoices')
                ->whereRaw('1 = 0'); // This will return no results

            return view('livewire.tax-report', [
                'franchises' => collect(),
                'insuranceCompanies' => collect(),
                'adjusters' => collect(),
                'appraisers' => collect(),
                'data' => $emptyQuery->paginate($this->perPage)
            ]);
        }
    }

    public function updatingFromDate()
    {
        $this->resetPage();
    }

    public function updatingToDate()
    {
        $this->resetPage();
    }

    public function updatingFranchise()
    {
        $this->resetPage();
    }

    public function updatingInsuranceCompany()
    {
        $this->resetPage();
    }

    public function updatingUnitType()
    {
        $this->resetPage();
    }

    public function updatingGlobalStatus()
    {
        $this->resetPage();
    }

    public function updatingPerPage()
    {
        $this->resetPage();
    }

    public function clearFilters()
    {
        $this->reset([
            'from_date',
            'to_date',
            'franchise',
            'insurance_company',
            'unit_type',
            'global_status'
        ]);
        $this->resetPage();
        $this->isLoading = false;
    }

    public function exportData()
    {
        // This method can be implemented later for CSV/Excel export
        $this->dispatch('show-message', 'Export functionality will be implemented soon.');
    }

    public function exportCsv(): StreamedResponse
    {
        $filename = 'tax-report-'.now()->format('Y-m-d_H-i-s').'.csv';

        // Rebuild the SAME query you use in render(), with the SAME filters
        $query = DB::table('assignment_invoices')
            ->join('assignments', 'assignments.id', '=', 'assignment_invoices.assignment_id')
            // ->join('tb_invoice_transactions', 'tb_invoice_transactions.invoiceID', '=', 'assignment_invoices.invoiceID')
            ->where('assignment_invoices.invoice_status', 'Paid');

        if (count($this->multiple_franchise) > 0) {
            $query->whereIn('assignments.franchise_id', $this->multiple_franchise);
        }

        // Date filters (gen_date — same as your render)
        if ($this->from_date && $this->to_date) {
            $query->whereBetween('assignment_invoices.received_date', [
                $this->from_date.' 00:00:00',
                $this->to_date.' 23:59:59',
            ]);
        } elseif ($this->from_date && !$this->to_date) {
            $query->where('assignment_invoices.received_date', '>=', $this->from_date.' 00:00:00');
        } elseif ($this->to_date && !$this->from_date) {
            $query->where('assignment_invoices.received_date', '<=', $this->to_date.' 23:59:59');
        }

        // Franchise filter
        if (!empty($this->franchise)) {
            $query->whereIn('assignments.franchise_id', explode(',', $this->franchise));
        }

        // Insurance Company filter
        if (!empty($this->insurance_company)) {
            $query->whereIn('assignments.insurance_company_id', explode(',', $this->insurance_company));
        }

        // Unit Type filter
        if (!empty($this->unit_type)) {
            $query->whereIn('assignments.unit_type_id', explode(',', $this->unit_type));
        }

        // Same constraints as render()
        $query->where('assignment_invoices.taxpercentage', '>', 0)
            ->orderBy('assignments.id', 'DESC');

        // For CSV, select explicit columns to avoid collisions
        $query->select([
            'assignments.id',
            'assignments.file_no',
            'assignments.franchise',
            'assignments.vehicle_city',
            'assignments.vehicle_state',
            'assignments.vehicle_postcode',
            'assignments.insurance_company_id',
            'assignments.unit_type_id',
            'assignment_invoices.id as invoiceID',
            'assignments.file_no as invoice_fileno',
            'assignment_invoices.invoice_amount',
            'assignment_invoices.taxpercentage',
            'assignment_invoices.amount_paid',
            'assignment_invoices.invoice_status',
            'assignment_invoices.gen_date',
            // 'tb_invoice_transactions.paymentDate',
        ]);

        return response()->streamDownload(function () use ($query) {
            $out = fopen('php://output', 'w');

            // UTF-8 BOM (so Excel opens it correctly)
            fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));

            // Header row
            fputcsv($out, [
                'Assignment ID',
                'File No',
                'Franchise ID',
                'Insurance Company ID',
                'Unit Type',
                'Invoice ID',
                'Invoice File No',
                'Invoice Amount',
                'Tax %',
                'Tax $',
                'Amount Paid',
                'Invoice Status',
                'Invoice Generated (gen_date)',
                'Payment Date',
                'City',
                'State',
                'Zip'
            ]);

            // Stream results (low memory)
            foreach ($query->cursor() as $row) {
                $taxAmount = round(($row->invoice_amount * $row->taxpercentage) / 100, 2);

                fputcsv($out, [
                    $row->id,
                    $row->file_no,
                    $row->franchise,
                    $row->insurance_company_id,
                    $row->unit_type,
                    $row->invoiceID,
                    $row->invoice_fileno,
                    $row->invoice_amount,
                    $row->taxpercentage,
                    $taxAmount,
                    $row->amount_paid,
                    $row->invoice_status,
                    $row->gen_date,
                    $row->paymentDate,
                    $row->vehicle_city,
                    $row->vehicle_state,
                    $row->vehicle_postcode,
                ]);
            }

            fclose($out);
        }, $filename, [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
        ]);
    }
}

