<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use Livewire\WithPagination;
use Illuminate\Http\Request;
use Carbon\Carbon;

class Summary extends Component
{
    use WithPagination;
    public $perPage = 25;

    public $totalBillAmount = 0;
    public $paidBillAmount = 0;
    public $unpaidBillAmount = 0;

    public $showgraph = 'thismonth';
    public $filter_month;
    public $filter_year;

    public $total = 0;
    public $paid = 0;
    public $unpaid = 0;

    public $daterange = [];


    public $chartAmounts = [];
    public $chartDates = [];
    public $chartDataCount = [];
    public $chartDataSuppCount = [];
    public $chartDataCompleteCount = [];
    public $Monthtomonthdata = [];
    public $monthtomonthpiechartData = [];

    public $chartmonthLabels = [];
    public $chartmonthFiles = [];
    public $chartmonthInvoice = [];
    public $chartmonthPaid = [];
    public $chartmonthUnpaid = [];
    public $chartmonthSupplement = [];
    public $multiple_franchise = [];
    public $completed_status_id;
    public $closed_status_id;

    public function mount(Request $request)
    {
        $tempRecord = GlobalStatus::whereRaw('LOWER(status) = ?', [strtolower('Completed')])->first();
        if ($tempRecord) {
            $this->completed_status_id = $tempRecord->id;
        }
        $tempRecord = GlobalStatus::whereRaw('LOWER(status) = ?', [strtolower('Closed')])->first();
        if ($tempRecord) {
            $this->closed_status_id = $tempRecord->id;
        }

        // If user is office admin, then limit the records to the multiple franchise only;
        if (auth()->user()->franchise != 0 && auth()->user()->franchise != '' && auth()->user()->franchise != null)
        {
            $franchises = auth()->user()->multiple_franchise;
            $this->multiple_franchise = $franchises 
                ? array_map('trim', explode(',', $franchises)) 
                : [];
            $this->multiple_franchise[] = auth()->user()->franchise;
            $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
        }

        $this->showgraph = $request->query('showgraph');
        $this->filter_month = $request->query('filter_month');
        $this->filter_year = $request->query('filter_year');
        if ($this->showgraph == null) {
            $this->showgraph = 'thismonth';
        }
        $this->fetchSummary();
        $this->fetchChartData($this->showgraph);
        $this->monthToMonthRecord();
    }



    public function monthToMonthRecord()
    {
        $query = DB::table('assignment_invoices')
            ->join('assignments', 'assignments.id', '=', 'assignment_invoices.assignment_id')
            ->where('assignment_invoices.invoice_status', '!=', 'Void')
            ->whereNotNull('assignment_invoices.gen_date');

        if (count($this->multiple_franchise) > 0) {
            $query = $query->whereIn('assignments.franchise_id', $this->multiple_franchise);
        }

            //->whereNull('supplement')
        $rawDataCollection =   $query->select(
                DB::raw('MONTH(assignment_invoices.gen_date) as month'),
                DB::raw('YEAR(assignment_invoices.gen_date) as year'),
                //DB::raw('COUNT(assignment_invoices.assignment_id) as total_files'), // counts multiple invoices
                DB::raw("COUNT(CASE WHEN assignments.supplement IS NULL OR assignments.supplement = '' THEN assignment_invoices.assignment_id END) as total_files"),
                DB::raw('COUNT(assignments.supplement) as supplement_count'),
                DB::raw('SUM(invoice_amount * (1 + COALESCE(taxpercentage, 0) / 100)) as total_invoice_amount'),
                DB::raw('SUM(assignment_invoices.amount_paid) as total_paid_amount'),
                DB::raw('SUM(assignment_invoices.invoice_amount - IFNULL(assignment_invoices.amount_paid, 0)) as total_unpaid_amount')

            )
            ->groupBy(DB::raw('YEAR(assignment_invoices.gen_date)'), DB::raw('MONTH(assignment_invoices.gen_date)'))
            ->orderBy(DB::raw('YEAR(assignment_invoices.gen_date)'), 'desc')
            ->orderBy(DB::raw('MONTH(assignment_invoices.gen_date)'), 'desc')
            ->limit(12)
            ->get();


        // 🔑 Re-index for easy lookup by year-month key
        $rawData = $rawDataCollection->keyBy(function ($item) {
            return $item->year . '-' . str_pad($item->month, 2, '0', STR_PAD_LEFT);
        });

        $this->Monthtomonthdata = [];
        $this->chartmonthLabels = collect();
        $this->chartmonthFiles = collect();
        $this->chartmonthInvoice = collect();
        $this->chartmonthPaid = collect();
        $this->chartmonthUnpaid = collect();
        $this->chartmonthSupplement = collect();

        $now = Carbon::now();

        for ($i = 0; $i < 12; $i++) {
            $date = $now->copy()->subMonths($i);
            $key = $date->format('Y-m');

            if (isset($rawData[$key])) {
                $entry = (array) $rawData[$key];
            } else {
                $entry = [
                    'month' => (int) $date->format('m'),
                    'year' => (int) $date->format('Y'),
                    'total_files' => 0,
                    'supplement_count' => 0,
                    'total_invoice_amount' => 0,
                    'total_paid_amount' => 0,
                    'total_unpaid_amount' => 0,
                ];
            }

            $this->Monthtomonthdata[] = $entry;

            if ($entry['total_files'] > 0) {
                // Append to chart data
                $this->chartmonthLabels->prepend(Carbon::createFromDate($entry['year'], $entry['month'], 1)->format('M Y'));
                $this->chartmonthFiles->prepend(round($entry['total_files'], 2));
                $this->chartmonthInvoice->prepend(round($entry['total_invoice_amount'], 2));
                $this->chartmonthPaid->prepend(round($entry['total_paid_amount'], 2));
                $this->chartmonthUnpaid->prepend(round($entry['total_unpaid_amount'], 2));
                $this->chartmonthSupplement->prepend(round($entry['supplement_count'], 2));
            }

        }




    }


    public function fetchSummary()
    {
        $franchiseFilter = session('franchise_filter'); // Optional: from session or request

        $queryBase = DB::table('assignment_invoices')
            ->leftJoin('assignments', 'assignments.id', '=', 'assignment_invoices.assignment_id');
        //->where('assignment_invoices.invoice_status', '!=', 'Void')->whereNull('assignments.supplement');

        if (count($this->multiple_franchise) > 0) {
            $queryBase = $queryBase->whereIn('assignments.franchise_id', $this->multiple_franchise);
        }

        $startDate = "";
        $endDate = "";
        $year = "";

        // Apply date filter based on showgraph
        if (in_array($this->showgraph, ['thismonth', 'lastmonth', 'customgraph'])) {
            if ($this->showgraph === 'customgraph' && $this->filter_month && $this->filter_year) {
                $startDate = Carbon::createFromDate($this->filter_year, $this->filter_month, 1)->startOfMonth();
            } elseif ($this->showgraph === 'lastmonth') {
                $startDate = now()->subMonth()->startOfMonth();
            } else {
                $startDate = now()->startOfMonth();
            }

            $endDate = $startDate->copy()->endOfMonth();
            $queryBase->whereBetween('gen_date', [$startDate->format('Y-m-d H:i:s'), $endDate->format('Y-m-d 23:59:59')]);

        } elseif (in_array($this->showgraph, ['thisyear', 'lastyear'])) {
            $year = $this->showgraph === 'lastyear' ? now()->subYear()->year : now()->year;
            $queryBase->whereYear('gen_date', $year);
        }


        $this->daterange[] = array("startDate" => $startDate, "endDate" => $endDate, "year" => $year);


        $this->totalBillAmount = (clone $queryBase)
            ->where('invoice_status', '!=', 'Void')->whereNotNull('gen_date')
            ->sum(DB::raw('invoice_amount * (1 + taxpercentage / 100)'));




        

        $this->unpaidBillAmount = (clone $queryBase)
            ->where('invoice_status', '!=', 'Void')
            ->whereNotNull('gen_date')
            ->where('assignment_invoices.invoice_amount', '>', 0)
            ->sum(DB::raw('(invoice_amount * (1 + COALESCE(taxpercentage, 0) / 100)) - COALESCE(amount_paid, 0)'));

        $this->paidBillAmount = (clone $queryBase)
            ->where('invoice_status', '=', 'Paid')
            ->sum('amount_paid');
    }


    public function fetchChartData($graphType = 'thismonth', $month = null, $year = null)
    {

        // Apply date filter based on showgraph
        if (in_array($this->showgraph, ['thismonth', 'lastmonth', 'customgraph'])) {
            if ($this->showgraph === 'customgraph' && $this->filter_month && $this->filter_year) {
                $startDate = Carbon::createFromDate($this->filter_year, $this->filter_month, 1)->startOfMonth();
            } elseif ($this->showgraph === 'lastmonth') {
                $startDate = now()->subMonth()->startOfMonth();
            } else {
                $startDate = now()->startOfMonth();
            }

            $endDate = $startDate->copy()->endOfMonth();

            while ($startDate->lte($endDate)) {
                $label = $startDate->format('d-M');
                $this->chartDates[] = $label;


                $this->chartAmounts[] = round((float) DB::table('assignment_invoices')
                    ->whereDate('gen_date', $startDate->format('Y-m-d'))->where('invoice_status', '!=', 'Void')
                    ->sum('invoice_amount'), 2);

                //                $this->chartDataCount[] = DB::table('assignments')
//                    ->whereDate('created_date', $startDate->format('Y-m-d'))
//                    ->count();

                $_query = DB::table('assignments');
                if (count($this->multiple_franchise) > 0) {
                    $_query = $_query->whereIn('assignments.franchise_id', $this->multiple_franchise);
                }
                $this->chartDataCount[] = $_query
                    ->whereDate('created_date', $startDate->format('Y-m-d'))
                    ->where(function ($query) {
                        $query->whereNull('supplement')
                            ->orWhere('supplement', '');
                    })
                    ->count();

                $_query = DB::table('assignments');
                if (count($this->multiple_franchise) > 0) {
                    $_query = $_query->whereIn('assignments.franchise_id', $this->multiple_franchise);
                }
                $this->chartDataSuppCount[] = $_query
                    ->whereDate('created_date', $startDate->format('Y-m-d'))
                    ->where('supplement', 'YES')
                    ->count();

                $_query = DB::table('assignments');
                if (count($this->multiple_franchise) > 0) {
                    $_query = $_query->whereIn('assignments.franchise_id', $this->multiple_franchise);
                }
                $this->chartDataCompleteCount[] = $_query
                    ->whereDate('complete_date', $startDate->format('Y-m-d'))
                    ->where(function ($query) {
                        $query->whereNull('supplement')
                            ->orWhere('supplement', '');
                    })
                    ->count();

                $startDate->addDay();
            }
        } elseif (in_array($graphType, ['thisyear', 'lastyear'])) {
            $selectedYear = $graphType === 'lastyear' ? now()->subYear()->year : now()->year;

            for ($m = 1; $m <= 12; $m++) {
                $label = Carbon::create($selectedYear, $m, 1)->format('M y');
                $this->chartDates[] = $label;

                $this->chartAmounts[] = round((float) DB::table('assignment_invoices')
                    ->whereYear('gen_date', $selectedYear)
                    ->whereMonth('gen_date', $m)
                    ->where('invoice_status', '!=', 'Void')
                    ->sum('invoice_amount'), 2);

                $_query = DB::table('assignments');
                if (count($this->multiple_franchise) > 0) {
                    $_query = $_query->whereIn('assignments.franchise_id', $this->multiple_franchise);
                }
                $this->chartDataCount[] = $_query
                    ->whereYear('created_date', $selectedYear)
                    ->whereMonth('created_date', $m)
                    ->where(function ($query) {
                        $query->whereNull('supplement')
                            ->orWhere('supplement', '');
                    })
                    ->count();

                // Count ONLY supplements
                $_query = DB::table('assignments');
                if (count($this->multiple_franchise) > 0) {
                    $_query = $_query->whereIn('assignments.franchise_id', $this->multiple_franchise);
                }
                $this->chartDataSuppCount[] = $_query
                    ->whereYear('created_date', $selectedYear)
                    ->whereMonth('created_date', $m)
                    ->whereNotNull('supplement')
                    ->where('supplement', '!=', '')
                    ->count();


                $_query = DB::table('assignments');
                if (count($this->multiple_franchise) > 0) {
                    $_query = $_query->whereIn('assignments.franchise_id', $this->multiple_franchise);
                }
                $this->chartDataCompleteCount[] = $_query
                    ->whereYear('complete_date', $selectedYear)
                    ->whereMonth('complete_date', $m)
                    ->count();
            }
        }
    }






    public function updatedShowgraph($value)
    {
        $this->fetchChartData($value, $this->filter_month, $this->filter_year);
    }

    public function updatedFilterMonth()
    {
        if ($this->showgraph === 'customgraph') {
            $this->fetchChartData($this->showgraph, $this->filter_month, $this->filter_year);
        }
    }

    public function updatedFilterYear()
    {
        if ($this->showgraph === 'customgraph') {
            $this->fetchChartData($this->showgraph, $this->filter_month, $this->filter_year);
        }
    }

    private function getAvgOfficeCycleTimes()
    {
        $averagecycletime = DB::table('assignments')
            ->join('franchise', 'assignments.franchise_id', '=', 'franchise.id')
            ->select(
                'franchise.company_name as officename',
                'franchise.franchisecode as officecode',
                DB::raw('AVG(DATEDIFF(assignments.complete_date, assignments.created_date)) as avg_cycle_days')
            );

        if (count($this->multiple_franchise) > 0) {
            $averagecycletime = $averagecycletime->whereIn('assignments.franchise_id', $this->multiple_franchise);
        }

        $averagecycletime = $averagecycletime
            ->whereIn('assignments.status_id', [$this->completed_status_id, $this->closed_status_id])
            ->whereNotNull('assignments.complete_date')
            ->where('assignments.complete_date', '!=', '')
            ->groupBy('assignments.franchise_id', 'franchise.company_name')
            ->orderBy('avg_cycle_days', 'asc')
            ->get();
        return $averagecycletime;
    }


    public function render()
    {
        return view('livewire.summary', [
            'totalBillAmount' => $this->totalBillAmount,
            'paidBillAmount' => $this->paidBillAmount,
            'unpaidBillAmount' => $this->unpaidBillAmount,
            'chartDates' => $this->chartDates,
            'chartAmounts' => $this->chartAmounts,
            'chartDataCount' => $this->chartDataCount,
            'chartDataSuppCount' => $this->chartDataSuppCount,
            'chartDataCompleteCount' => $this->chartDataCompleteCount,
            'showGraph' => $this->showgraph,
            'filterMonth' => $this->filter_month,
            'filterYear' => $this->filter_year,
            'daterange' => $this->daterange,
            'Monthtomonthdata' => $this->Monthtomonthdata,
            'chartmonthLabels' => $this->chartmonthLabels,
            'chartmonthFiles' => $this->chartmonthFiles,
            'chartmonthInvoice' => $this->chartmonthInvoice,
            'chartmonthPaid' => $this->chartmonthPaid,
            'chartmonthUnpaid' => $this->chartmonthUnpaid,
            'chartmonthSupplement' => $this->chartmonthSupplement,
            'averagecycletime' => $this->getAvgOfficeCycleTimes()
        ]);
    }
}

