<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use App\Models\UnitType;
use Livewire\WithPagination;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ReportStageLifecycle extends Component
{

    use WithPagination;
    public $perPage = 20;
    public $from_date;
    public $to_date;
    public $franchise_id;
    public $insurance_company_id;
    public $adjuster_id;
    public $appraiser_id;
    public $unit_type = '';
    public $global_status = '';
    protected $paginationTheme = 'bootstrap';
    public $multiple_franchise = [];

    public function mount(Request $request)
    {
        // If user is office admin, then limit the records to the multiple franchise only;
        if (auth()->user()->franchise != 0 && auth()->user()->franchise != '' && auth()->user()->franchise != null)
        {
            $franchises = auth()->user()->multiple_franchise;
            $this->multiple_franchise = $franchises 
                ? array_map('trim', explode(',', $franchises)) 
                : [];
            $this->multiple_franchise[] = auth()->user()->franchise;
            $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
        }

        $this->from_date = $request->query('from_date');
        $this->to_date = $request->query('to_date');
        $this->franchise_id = $request->query('franchise_id');
        $this->insurance_company_id = $request->query('insurance_company_id');
        $this->adjuster_id = $request->query('adjuster_id');
        $this->appraiser_id = $request->query('appraiser_id');
        $this->selectedReport = $request->query('selectedReport');
        $this->unit_type = $request->query('unit_type');
        $this->global_status = $request->query('global_status');
    }

    public function formatHours($hours)
    {
        $days = floor($hours / 24);
        $weeks = floor($days / 7);
        $remainingDays = $days % 7;
        $remainingHours = round($hours % 24);

        if ($weeks > 0) {
            return "{$weeks} wk {$remainingDays} d {$remainingHours} hr";
        } elseif ($days > 0) {
            return "{$days} d {$remainingHours} hr";
        } else {
            return "{$remainingHours} hr";
        }
    }


    public function updating($property)
    {
        // Reset to page 1 on any filter change
        $this->resetPage();

    }

    public function search()
    {
        $this->resetPage();
    }



    public function render()
    {

        $query = DB::table('assignments as a')
            ->select('a.*')
            ->where('a.file_no', '!=', '');
        
        if (count($this->multiple_franchise) > 0) {
            $query->whereIn('a.franchise_id', $this->multiple_franchise);
        }

        // Apply filters
        if ($this->from_date) {
            $query->where('a.created_date', '>=', $this->from_date . ' 00:00:00');
        }

        if ($this->to_date) {
            $query->where('a.created_date', '<=', $this->to_date . ' 23:59:59');
        }

        if ($this->franchise_id) {
            $franchise_ids = explode(',', $this->franchise_id);
            $query->whereIn('a.franchise_id', $franchise_ids);
        }

        if ($this->insurance_company_id) {
            $insurance_company_ids = explode(',', $this->insurance_company_id);
            $query->whereIn('a.insurance_company_id', $insurance_company_ids);
        }

        if ($this->adjuster_id) {
            $adjuster_ids = explode(',', $this->adjuster_id);
            $query->whereIn('a.adjuster_id', $adjuster_ids);
        }

        if ($this->appraiser_id) {
            $appraiser_ids = explode(',', $this->appraiser_id);
            $query->whereIn('a.appraiser_id', $appraiser_ids);
        }

        if (!empty($this->unit_type)) 
        {
            $unit_type_ids = explode(',', $this->unit_type);
            // $unit_type_ids = array_map(function($id) {
            //     return "'$id'";
            // }, $unit_type_ids);
            $query->whereIn('a.unit_type_id', $unit_type_ids);
        }

        if (!empty($this->global_status)) 
        {
            $global_status_ids = explode(',', $this->global_status);
            // $global_status_ids = array_map(function($id) {
            //     return "'$id'";
            // }, $global_status_ids);
            $query->whereIn('a.status_id', $global_status_ids);
        }

        $assignments = $query->orderBy('a.created_date', 'desc')->paginate($this->perPage);

        // Loop and calculate lifecycle times
        foreach ($assignments as $assignment) {
            $logs = DB::table('assignment_logs')
                ->where('assignment_id', $assignment->id)
                ->orderBy('created_at')
                ->get();

            $created = strtotime($assignment->created_date);
            $dispatched = $submitted = $audit = $completed = 0;

            foreach ($logs as $log) {
                if (str_contains($log->activity, 'Assignment Dispatched') && $dispatched == 0)
                    $dispatched = strtotime($log->created_at);
                if (str_contains($log->activity, 'Appraiser Submit Assignment Completed') && $submitted == 0)
                    $submitted = strtotime($log->created_at);
                if (str_contains($log->activity, 'Assignment Passed Audit') && $audit == 0)
                    $audit = strtotime($log->created_at);
                if (str_contains($log->activity, 'Assignment Completed') && $completed == 0)
                    $completed = strtotime($log->created_at);
            }


            $assignment->unassigned = $this->formatHours(max(0, round((($dispatched ?? $created) - $created) / 3600, 2)));
            $assignment->assigned = $this->formatHours(max(0, round((($submitted ?? $created) - ($dispatched ?? $created)) / 3600, 2)));
            $assignment->file_submitted = $this->formatHours(max(0, round((($audit ?? $created) - ($submitted ?? $created)) / 3600, 2)));
            $assignment->passed_audit = $this->formatHours(max(0, round((($completed ?? $created) - ($audit ?? $created)) / 3600, 2)));

        }


        $adjusters = DB::table('users')->where('role', '4')->get();
        $franchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->get();
        $insuranceCompanies = DB::table('insurance_company')->orderBy('company_name', 'asc')->get();

        return view('livewire.report-stage-lifecycle', compact('assignments', 'franchises', 'insuranceCompanies', 'adjusters'))->layout('layouts.app');
        ;

    }
}

