<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use App\Models\UnitType;
use Livewire\WithPagination;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ReportAssignmentPerday extends Component
{
    use WithPagination;

    public $from_date;
    public $to_date;
    public $franchise_id;
    public $insurance_company_id;
    public $adjuster_id;
    public $appraiser_id;
    public $unit_type = '';
    public $global_status = '';
    public $multiple_franchise = [];

    protected $paginationTheme = 'bootstrap';
    public $chartDates = [];
    public $chartCounts = [];
    public $total_count = 0;

    public function mount(Request $request)
    {
        // If user is office admin, then limit the records to the multiple franchise only;
        if (auth()->user()->franchise != 0 && auth()->user()->franchise != '' && auth()->user()->franchise != null) {
            $franchises = auth()->user()->multiple_franchise;
            $this->multiple_franchise = $franchises
                ? array_map('trim', explode(',', $franchises))
                : [];
            $this->multiple_franchise[] = auth()->user()->franchise;
            $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
        }

        $this->from_date = $request->query('from_date');
        $this->to_date = $request->query('to_date');
        $this->franchise_id = $request->query('franchise_id');
        $this->insurance_company_id = $request->query('insurance_company_id');
        $this->adjuster_id = $request->query('adjuster_id');
        $this->appraiser_id = $request->query('appraiser_id');
        $this->unit_type = $request->query('unit_type');
        $this->global_status = $request->query('global_status');
        $this->generateReport();

    }

    public function updated($property)
    {
        if (
            in_array($property, [
                'from_date',
                'to_date',
                'franchise_id',
                'insurance_company_id',
                'adjuster_id'
            ])
        ) {
            $this->resetPage(); // reset pagination when filters change
            $this->generateReport();
        }
    }

    public function generateReport()
    {
        $this->chartDates = [];
        $this->chartCounts = [];
        $this->total_count = 0;

        $start = Carbon::parse($this->from_date);
        $end = Carbon::parse($this->to_date);
        $days = $end->diffInDays($start);

        for ($i = 0; $i <= $days; $i++) {
            $startdate = $start->copy()->addDays($i)->startOfDay();
            $enddate = $start->copy()->addDays($i)->endOfDay();
            $label = $startdate->format('d-M');

            $query = DB::table('assignments')
                ->whereBetween('created_date', [$startdate, $enddate]);

            if ($this->from_date) {
                $query->where('created_date', '>=', $this->from_date . ' 00:00:00');
            }

            if ($this->to_date) {
                $query->where('created_date', '<=', $this->to_date . ' 23:59:59');
            }

            if ($this->franchise_id) {
                $franchise_ids = explode(',', $this->franchise_id);
                $query->whereIn('franchise_id', $franchise_ids);
            }

            if ($this->insurance_company_id) {
                $insurance_company_ids = explode(',', $this->insurance_company_id);
                $query->whereIn('insurance_company_id', $insurance_company_ids);
            }

            if ($this->adjuster_id) {
                $adjuster_ids = explode(',', $this->adjuster_id);
                $query->whereIn('adjuster_id', $adjuster_ids);
            }

            if ($this->appraiser_id) {
                $appraiser_ids = explode(',', $this->appraiser_id);
                $query->whereIn('appraiser_id', $appraiser_ids);
            }

            if ($this->unit_type) {
                $unit_type_ids = explode(',', $this->unit_type);
                $query->whereIn('unit_type_id', $unit_type_ids);
            }

            if ($this->global_status) {
                $global_status_ids = explode(',', $this->global_status);
                $query->whereIn('status_id', $global_status_ids);
            }

            if (count($this->multiple_franchise) > 0) {
                $query->whereIn('franchise_id', $this->multiple_franchise);
            }

            $count = $query->count();
            if ($count > 0) {
                $this->chartDates[] = $label;
                $this->chartCounts[] = $count;
            }


            $this->total_count += $count;
        }
    }


    public function render()
    {

        $adjusters = DB::table('users')->where('role', '4')->get();
        if (count($this->multiple_franchise) > 0) {
            $franchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->whereIn('id', $this->multiple_franchise)->get();
        } else {
            $franchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->get();
        }
        $insuranceCompanies = DB::table('insurance_company')->orderBy('company_name', 'asc')->get();

        return view('livewire.report-assignment-perday', compact('franchises', 'insuranceCompanies', 'adjusters'))->layout('layouts.app');
        ;
    }
}

