<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use App\Models\UnitType;
use Livewire\WithPagination;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ReportAssignmentCycletime extends Component
{
    
    use WithPagination;

    public $from_date;
    public $to_date;
    public $franchise_id;
    public $insurance_company_id;
    public $adjuster_id;
    public $appraiser_id;
    public $unit_type = '';
    public $global_status = '';
    public $perPage = 25;

    protected $paginationTheme = 'bootstrap';
    public $chartDates = [];
    public $chartCounts = [];
    public $total_count = 0;
    public $multiple_franchise = [];

    public function mount(Request $request)
    {
        // If user is office admin, then limit the records to the multiple franchise only;
        if (auth()->user()->franchise != 0 && auth()->user()->franchise != '' && auth()->user()->franchise != null)
        {
            $franchises = auth()->user()->multiple_franchise;
            $this->multiple_franchise = $franchises 
                ? array_map('trim', explode(',', $franchises)) 
                : [];
            $this->multiple_franchise[] = auth()->user()->franchise;
            $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
        }

        $this->from_date = $request->query('from_date');
        $this->to_date = $request->query('to_date');
        $this->franchise_id = $request->query('franchise_id');
        $this->insurance_company_id = $request->query('insurance_company_id');
        $this->adjuster_id = $request->query('adjuster_id');
        $this->appraiser_id = $request->query('appraiser_id');
        $this->unit_type = $request->query('unit_type');
        $this->global_status = $request->query('global_status');

    }

    public function updated($property)
    {
        if (
            in_array($property, [
                'from_date',
                'to_date',
                'franchise_id',
                'insurance_company_id',
                'adjuster_id'
            ])
        ) {
            $this->resetPage(); // reset pagination when filters change
        }
    }

    public function render()
    {
        $query = DB::table('assignments as a')
            ->whereNotNull('a.complete_date')
            ->where('a.file_no', '!=', '');
            
        if (count($this->multiple_franchise) > 0) {
            $query->whereIn('a.franchise_id', $this->multiple_franchise);
        }

        if ($this->from_date) {
            $query->where('a.created_date', '>=', $this->from_date . ' 00:00:00');
        }

        if ($this->to_date) {
            $query->where('a.created_date', '<=', $this->to_date . ' 23:59:59');
        }

        $filters = [
            'a.insurance_company_id' => $this->insurance_company_id,
            'a.franchise_id' => $this->franchise_id,
            'a.adjuster_id' => $this->adjuster_id,
            'a.appraiser_id' => $this->appraiser_id,
            'a.unit_type_id' => $this->unit_type,
            'a.status_id' => $this->global_status,
        ];

        foreach ($filters as $column => $value) {
            if (!empty($value)) {
                $value = explode(',', $value);
                $query->whereIn($column, $value);
            }
        }


        $clonedQuery = clone $query;
        $average = $clonedQuery->selectRaw("AVG(TIMESTAMPDIFF(SECOND, a.created_date, a.complete_date)) as avg_cycle_time")->first();

        $records = $query
            ->orderBy('a.created_date', 'desc')
            ->paginate($this->perPage);

        // Calculate cycle time for each record
        foreach ($records as $record) {
            if (!empty($record->created_date) && !empty($record->complete_date)) {
                $record->cycle_time = $this->formatCycleTime(
                    strtotime($record->complete_date) - strtotime($record->created_date)
                );
            } else {
                $record->cycle_time = '-';
            }
        }

        // Show blank if no average value
        $avgCycleTime = ($average && $average->avg_cycle_time !== null)
            ? $this->formatCycleTime($average->avg_cycle_time)
            : '';

        return view('livewire.report-assignment-cycletime', [
            'records' => $records,
            'avgCycleTime' => $avgCycleTime,
        ])->layout('layouts.app');
    }

    private function formatCycleTime($seconds)
    {
        $seconds = abs($seconds);
        $days = floor($seconds / 86400);
        $hours = floor(($seconds % 86400) / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        return "{$days}d {$hours}h {$minutes}m";
    }
}

