<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use App\Models\UnitType;
use Livewire\WithPagination;
use Illuminate\Http\Request;
use Carbon\Carbon;

class Receivables extends Component
{
    use WithPagination;

    public $franchise, $insurance, $adjuster;
    public $status = '';
    public $allFranchises = [], $allInsurances = [], $allAdjusters = [];
    protected $paginationTheme = 'bootstrap';
    public $termname = 'all'; // default selection
    public $term = 'term'; // default value
    public $searchbyinvoiceno = '';
    public $searchbyfileno = '';
    public $fromDate = '';
    public $toDate = '';
    public $specificDate = '';
    public $showBulkPaymentModal = false;

    public $invoices_selected = [];
    public $bulk_date_received = '';
    public $bulk_payment_amount = '';
    public $bulk_payment_note = '';
    public $perPage = 25;
    
    public $recordIds = [];
    public $selectAll = false;
    public $unit_types = [];
    public $unit_type = '';
    public $global_statuses = [];
    public $global_status = '';
    public $multiple_franchise = [];
    public $assignment_closed_id;

    public function updated($property, $value)
    {
        \Log::info('updated property: ' . $property . ' value: ' . json_encode($value));
    }
    
    public function testLivewire()
    {
        \Log::info('=== LIVEWIRE TEST BUTTON CLICKED ===');
        session()->flash('message', 'Livewire is working!');
    }

    // Computed properties for field visibility
    public function getShowTermClassProperty()
    {
        \Log::info('getShowTermClassProperty called, term = ' . $this->term);
        return $this->term === 'term';
    }

    public function getShowCustomDateRangeClassProperty()
    {
        \Log::info('getShowCustomDateRangeClassProperty called, term = ' . $this->term);
        return $this->term === 'custom';
    }

    public function getShowSpecificDateClassProperty()
    {
        \Log::info('getShowSpecificDateClassProperty called, term = ' . $this->term);
        return $this->term === 'specific';
    }

    public function setTerm($value)
    {
        \Log::info('setTerm called with value: ' . $value);
        $this->term = $value;
        $this->dispatch('term-updated', term: $value);
    }

    /*
    public function updatedSelectAll($value)
    {
        if ($value) {
            $this->invoices_selected = $this->records->pluck('invoiceID')->toArray();
        } else {
            $this->invoices_selected = [];
        }
    }
    */

    public function updatedSelectAll($value)
    {
        \Log::info('updatedSelectAll called with value: ' . $value);
        \Log::info('recordIds count: ' . count($this->recordIds));
        \Log::info('recordIds: ' . implode(', ', $this->recordIds));
        
        if ($value && !empty($this->recordIds)) {
            $this->invoices_selected = $this->recordIds;
            \Log::info('Selected all invoices: ' . count($this->invoices_selected));
        } else {
            $this->invoices_selected = [];
            \Log::info('Cleared all selections');
        }
        
        // Calculate bulk payment amount
        $this->calculateBulkPaymentAmount();
    }
    
    public function toggleSelectAll($value)
    {
        \Log::info('toggleSelectAll called with value: ' . $value);
        $this->selectAll = $value;
    }

    public function updatedInvoicesSelected()
    {
        // This method will be called whenever the invoices_selected array is updated
        // You can add any additional logic here if needed
        \Log::info('Selected invoices: ' . implode(', ', $this->invoices_selected));

        // Update selectAll checkbox state based on current selection
        $this->updateSelectAllState();

        $this->calculateBulkPaymentAmount();
    }

    public function updateSelectAllState()
    {
        if (empty($this->recordIds)) {
            $this->selectAll = false;
            return;
        }

        try {
            $currentPageIds = $this->recordIds;
            $selectedIds = $this->invoices_selected;

            // Check if all current page IDs are in the selected array
            $allSelected = !empty($currentPageIds) && count(array_intersect($currentPageIds, $selectedIds)) === count($currentPageIds);
            
            $this->selectAll = $allSelected;
        } catch (\Exception $e) {
            $this->selectAll = false;
        }
    }

    public function toggleInvoiceSelection($invoiceId)
    {
        \Log::info('toggleInvoiceSelection called with invoiceId: ' . $invoiceId);
        
        if (in_array($invoiceId, $this->invoices_selected)) {
            // Remove from selection
            $this->invoices_selected = array_values(array_diff($this->invoices_selected, [$invoiceId]));
            \Log::info('Removed invoice ' . $invoiceId . ' from selection');
        } else {
            // Add to selection
            $this->invoices_selected[] = $invoiceId;
            \Log::info('Added invoice ' . $invoiceId . ' to selection');
        }
        
        \Log::info('Current selected invoices: ' . implode(', ', $this->invoices_selected));

        // Update select all state
        $this->updateSelectAllState();
        
        // Calculate bulk payment amount
        $this->calculateBulkPaymentAmount();
    }

    public function clearSelection()
    {
        $this->invoices_selected = [];
        $this->selectAll = false;
        $this->recordIds = [];
    }
    public function mount()
    {
        \Log::info('Mounted... ');

        // If user is office admin, then limit the records to the multiple franchise only;
        if (auth()->user()->franchise != 0 && auth()->user()->franchise != '' && auth()->user()->franchise != null)
        {
            $franchises = auth()->user()->multiple_franchise;
            $this->multiple_franchise = $franchises 
                ? array_map('trim', explode(',', $franchises)) 
                : [];
            $this->multiple_franchise[] = auth()->user()->franchise;
            $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
        }

        $tempRecord = GlobalStatus::whereRaw('LOWER(status) = ?', [strtolower('Closed')])->first();
        if (!$tempRecord) {
            $tempRecord = GlobalStatus::create([
                'status' => 'Closed',
                'type' => 'For All',
                'detail' => 'Closed',
                'sequence' => '999',
                'friendly_name' => 'Closed',
            ]);
        }

        $this->assignment_closed_id = $tempRecord->id;


        $this->allAdjusters = DB::table('users')->where('role', '4')->get();
        if (count($this->multiple_franchise) > 0) {
            $this->allFranchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->whereIn('id', $this->multiple_franchise)->get();
        } else {
            $this->allFranchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->get();
        }
        $this->allInsurances = DB::table('insurance_company')->orderBy('company_name', 'asc')->get();
        
        // Debug logging for insurance companies
        \Log::info('allFranchises loaded: ' . $this->allFranchises->count());
        if ($this->allInsurances->count() > 0) {
            $first = $this->allInsurances->first();
            \Log::info('First insurance company: ' . json_encode($first));
            \Log::info('First insurance company fields: ' . implode(', ', array_keys((array)$first)));
            \Log::info('id field: ' . ($first->id ?? 'NOT FOUND'));
            \Log::info('company_name field: ' . ($first->company_name ?? 'NOT FOUND'));
        }
        
        $this->termname = request()->get('termname', 'all'); // optional: initialize from URL
        $this->term = request()->get('term', 'term'); // initialize from URL/request if needed
        $this->searchbyinvoiceno = request()->get('searchbyinvoiceno', '');
        $this->searchbyfileno = request()->get('searchbyfileno', '');
        $this->fromDate = request()->get('fromDate', '');
        $this->toDate = request()->get('toDate', '');
        $this->bulk_date_received = date('Y-m-d');
        $this->selectAll = false;
        $this->invoices_selected = [];
        $this->recordIds = [];
        $this->unit_types = UnitType::all();
        $this->global_statuses = GlobalStatus::all();
    }

    public function refreshOptions()
    {
        // Refresh the options data
        $this->allInsurances = DB::table('insurance_company')->orderBy('company_name', 'asc')->get();
        $this->allAdjusters = DB::table('users')->where('role', '4')->get();
        if (count($this->multiple_franchise) > 0) {
            $this->allFranchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->whereIn('id', $this->multiple_franchise)->get();
        } else {
            $this->allFranchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->get();
        }
        $this->unit_types = UnitType::all();
        $this->global_statuses = GlobalStatus::all();

        \Log::info('Options refreshed - Franchises: ' . $this->allFranchises->count());

        \Log::info('Options refreshed - Insurance companies: ' . $this->allInsurances->count());
        if ($this->allInsurances->count() > 0) {
            $first = $this->allInsurances->first();
            \Log::info('First insurance company after refresh: ' . json_encode($first));
            \Log::info('id field after refresh: ' . ($first->id ?? 'NOT FOUND'));
            \Log::info('company_name field after refresh: ' . ($first->company_name ?? 'NOT FOUND'));
        }
    }


    /*
    public function updating($property)
    {
        \Log::info('updating '.$property);
        // Reset to page 1 on any filter change
        // $this->resetPage();

        // Reset selection when filters change
        if (in_array($property, ['franchise', 'insurance', 'adjuster', 'status', 'searchbyinvoiceno', 'searchbyfileno', 'fromDate', 'toDate', 'termname', 'specificDate'])) {
            $this->invoices_selected = [];
            $this->selectAll = false;
            $this->recordIds = [];
        }
    }
    */

    public function search()
    {
        $this->resetPage();
        // Refresh options after search to ensure multiselect has latest data
        $this->refreshOptions();
        
        // Force a re-render of the multiselect component
        $this->dispatch('multiselect-refresh');
    }


    public function render()
    {
        // Base Query
        $query = DB::table('assignment_invoices')
            ->leftJoin('assignments', 'assignments.id', '=', 'assignment_invoices.assignment_id')
            ->leftJoin('insurance_company', 'assignments.insurance_company_id', '=', 'insurance_company.id')
            ->leftJoin('franchise', 'assignments.franchise_id', '=', 'franchise.id')
            ->select(
                'assignments.*',
                'assignment_invoices.id as invoiceID',
                'franchise.company_name as officename',
                'insurance_company.company_name as insurancecompanyname',
                'assignment_invoices.invoice_amount as finaltotal',
                'assignment_invoices.invoiceno',
                'assignment_invoices.gen_date',
                'assignment_invoices.invoice_amount',
                'assignment_invoices.taxpercentage',
                'assignment_invoices.amount_paid',
                'insurance_company.paying_type'
            );
        $query->where('assignment_invoices.invoice_amount', '>', 0);
        $query->whereNotNull('assignment_invoices.gen_date');
        $query->where('assignment_invoices.invoice_status', '!=', 'Void');

        if (count($this->multiple_franchise) > 0) {
            $query->whereIn('assignments.franchise_id', $this->multiple_franchise);
        }


        // Apply filters
        if(!empty($this->unit_type))
        {
            $unit_type_ids = explode(',', $this->unit_type);
            // $unit_type_ids = array_map(function($id) {
            //     return "'$id'";
            // }, $unit_type_ids);
            // \Log::info('unit_type_ids: '.implode(', ', $unit_type_ids));

            $query->whereIn('assignments.unit_type_id', $unit_type_ids);
        }

        if(!empty($this->global_status))
        {
            $global_status_ids = explode(',', $this->global_status);
            $query->whereIn('assignments.status_id', $global_status_ids);
        }
        
        if(!empty($this->franchise))
        {
            $franchise_ids = explode(',', $this->franchise);
            $query->whereIn('assignments.franchise_id', $franchise_ids);
        }
        
        if(!empty($this->insurance))
        {
            $insurance_ids = explode(',', $this->insurance);
            $query->whereIn('assignments.insurance_company_id', $insurance_ids);
        }
        
        if(!empty($this->adjuster))
        {
            \Log::info('adjuster in filter: '.$this->adjuster);
            $adjuster_ids = explode(',', $this->adjuster);
            $query->whereIn('assignments.adjuster_id', $adjuster_ids);
        }

        if ($this->status === 'paid') {
            $query->where('assignment_invoices.mark_paid', 1);
        } elseif ($this->status === 'unpaid') {
            $query->whereRaw('ROUND((invoice_amount + (invoice_amount * COALESCE(taxpercentage, 0) / 100)), 2) > ROUND(COALESCE(amount_paid, 0), 2)');

        }

        if (!empty($this->searchbyinvoiceno)) {
            // Explode by comma, trim whitespace, and remove empty values
            $invoiceNumbers = collect(explode(',', $this->searchbyinvoiceno))
                ->map(fn($item) => preg_replace('/\s+/', '', $item))
                ->filter()
                ->unique()
                ->toArray();

            if (count($invoiceNumbers) > 0) {
                $query->where(function ($q) use ($invoiceNumbers) {
                    foreach ($invoiceNumbers as $invoiceNo) {
                        $q->orWhere('assignment_invoices.invoiceno', $invoiceNo);
                    }
                });
            }
        }


        if (!empty($this->searchbyfileno)) {
            $keywords = collect(explode(',', $this->searchbyfileno))
                ->map(fn($item) => trim(preg_replace('/\s+/', '', $item)))
                ->filter()
                ->unique()
                ->toArray();

            if (count($keywords) > 0) {
                $query->where(function ($q) use ($keywords) {
                    foreach ($keywords as $keyword) {
                        $q->orWhere(function ($subQuery) use ($keyword) {
                            $subQuery->where('file_no', 'like', "%$keyword%")
                                ->orWhere('claim_number', 'like', "%$keyword%")
                                ->orWhere('policy_number', 'like', "%$keyword%")
                                ->orWhere('business_name', 'like', "%$keyword%");
                        });
                    }
                });
            }
        }

        if (!empty($this->fromDate) && !empty($this->toDate)) {
            \Log::info('fromDate and toDate in filter: '.$this->fromDate.' '.$this->toDate);

            $from = Carbon::parse($this->fromDate)->startOfDay();
            $to = Carbon::parse($this->toDate)->endOfDay();

            $query->whereBetween('assignment_invoices.gen_date', [$from, $to]);
        } elseif (!empty($this->termname) && $this->term === 'term') {
            \Log::info('termname in filter: '.$this->termname);

            switch ($this->termname) {
                case '30':
                    $start = now()->subDays(30)->startOfDay();
                    $end = now()->endOfDay();
                    $query->whereBetween('assignment_invoices.gen_date', [$start, $end]);
                    break;

                case '31':
                    $end = now()->subDays(31)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;

                case '60':
                    $start = now()->subDays(60)->startOfDay();
                    $end = now()->subDays(30)->endOfDay();
                    $query->whereBetween('assignment_invoices.gen_date', [$start, $end]);
                    break;

                case '61':
                    $end = now()->subDays(61)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;

                case '90':
                    $start = now()->subDays(90)->startOfDay();
                    $end = now()->subDays(60)->endOfDay();
                    $query->whereBetween('assignment_invoices.gen_date', [$start, $end]);
                    break;

                case '91':
                    $end = now()->subDays(91)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;

                case '121':
                    $end = now()->subDays(121)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;
            }
        }
        elseif (!empty($this->specificDate) && $this->term === 'specific') {
            \Log::info('specificDate in filter: '.$this->specificDate);

            $specificDate = Carbon::parse($this->specificDate)->startOfDay();
            $query->whereDate('assignment_invoices.gen_date', $specificDate);
        }
        
        // Calculate total amount before pagination
        $totalinvoiceAmount = $query->sum('assignment_invoices.invoice_amount');

        $totalTaxAmount = $query->sum(DB::raw('assignment_invoices.invoice_amount * (assignment_invoices.taxpercentage / 100)'));

        // Calculate total amount before pagination
        $totalpaidAmount = $query->sum('assignment_invoices.amount_paid');
        //$qry = $query->orderByDesc('assignment_invoices.gen_date')->toSql();
        $qry = "";

        if (!empty($this->perPage) && $this->perPage === 'all') {
            $records = $query->orderByDesc('assignment_invoices.gen_date')->get();
            $this->recordIds = $records->pluck('invoiceID')->toArray();
        } else {
            $perPage = !empty($this->perPage) ? (int) $this->perPage : 25;
            $records = $query->orderByDesc('assignment_invoices.gen_date')->paginate($perPage);
            $this->recordIds = $records->pluck('invoiceID')->toArray();
        }

        \Log::info('query: '.$query->toSql());

        //$records = $query->orderByDesc('assignment_invoices.gen_date')->paginate(25);

        // Update select all state after records are loaded
        $this->updateSelectAllState();

        return view('livewire.receivables', compact('records', 'totalinvoiceAmount', 'totalpaidAmount', 'qry', 'totalTaxAmount'));

    }



    public function openBulkPaymentModal()
    {
        \Log::info('openBulkPaymentModal... ');
        $this->calculateBulkPaymentAmount();
        $this->showBulkPaymentModal = true;
    }

    public function closeBulkPaymentModal()
    {
        $this->showBulkPaymentModal = false;
    }

    public function processBulkPayment()
    {
        // Validate the form data
        // $this->validate([
        //     'bulk_date_received' => 'required|date',
        //     'bulk_payment_amount' => 'required|numeric|min:0',
        //     'bulk_payment_note' => 'required|string|min:1',
        // ]);

        foreach ($this->invoices_selected as $invoiceID) {

            \Log::info('Processing invoice: ' . $invoiceID);

            
            $invoice = DB::table('assignment_invoices')->where('id', $invoiceID)->first();


            $invoice_amount = $invoice->invoice_amount;
            $taxpercentage = $invoice->taxpercentage;
            $amount_paid = $invoice->amount_paid;
            $totaldue = $invoice->invoice_amount;

            if (count($this->invoices_selected) > 1) {
                $totaldue = $invoice_amount + ($invoice_amount * $taxpercentage / 100);
            } else {
                $totaldue = $this->bulk_payment_amount;
            }
            

            if ($amount_paid == '' || $amount_paid == NULL) {
                $amount_paid = 0;
            }
            if ($totaldue == '' || $totaldue == NULL) {
                $totaldue = 0;
            }
            if ($invoice_amount == '' || $invoice_amount == NULL) {
                $invoice_amount = 0;
            }
    
            $outstanding = $invoice_amount - $amount_paid;

            // Update the invoice using query builder
            DB::table('assignment_invoices')->where('id', $invoiceID)->update([
                'mark_paid' => 1,
                'amount_paid' => $totaldue,
                'invoice_status' => 'Paid',
                'received_date' => $this->bulk_date_received
            ]);

            // Insert into tb_invoice_transactions
            // DB::table('tb_invoice_transactions')->insert([
            //     'invoiceID' => $invoiceID,
            //     'amount' => $outstanding,
            //     'paymentDate' => $this->bulk_date_received,
            //     'note' => $this->bulk_payment_note,
            //     'userID' => auth()->user()->id
            // ]);

            // Add activity log
            DB::table('assignment_logs')->insert([
                'assignment_id' => $invoice->assignment_id,
                'activity' => "Payment Collected - " . $outstanding,
                'user_id' => auth()->user()->id,
                'user_ip' => request()->ip(),
                'description' => ''
            ]);

            // update assignment status to closed
            DB::table('assignments')->where('id', $invoice->assignment_id)->update([
                'status_id' => $this->assignment_closed_id
            ]);

            // Add activity log
            DB::table('assignment_logs')->insert([
                'assignment_id' => $invoice->assignment_id,
                'activity' => "Assignment Closed",
                'user_id' => auth()->user()->id,
                'user_ip' => request()->ip(),
                'description' => ''
            ]);
            

        }
        
        
        $this->closeBulkPaymentModal();
        
        // Reset the form fields
        $this->bulk_date_received = '';
        $this->bulk_payment_amount = '';
        $this->bulk_payment_note = '';
        
        // You can add a success message here
        session()->flash('message', 'Bulk payment processed successfully!');
    }



    public function exportToCSV()
    {
        // Build the same query as in render() method but without pagination
        $query = DB::table('assignment_invoices')
            ->leftJoin('assignments', 'assignments.id', '=', 'assignment_invoices.assignment_id')
            ->leftJoin('insurance_company', 'assignments.insurance_company_id', '=', 'insurance_company.id')
            ->leftJoin('franchise', 'assignments.franchise_id', '=', 'franchise.id')
            ->select(
                'assignments.*',
                'assignment_invoices.id as invoiceID',
                'franchise.company_name as officename',
                'insurance_company.company_name as insurancecompanyname',
                'assignment_invoices.invoice_amount as finaltotal',
                'assignment_invoices.invoiceno',
                'assignment_invoices.gen_date',
                'assignment_invoices.invoice_amount',
                'assignment_invoices.taxpercentage',
                'assignment_invoices.amount_paid',
                'insurance_company.paying_type'
            );

        $query->where('assignment_invoices.invoice_amount', '>', 0);
        $query->whereNotNull('assignment_invoices.gen_date');
        $query->where('assignment_invoices.invoice_status', '!=', 'Void');

        // Apply filters
        if ($this->franchise)
            $query->where('assignments.franchise_id', $this->franchise);
        if ($this->insurance)
            $query->where('assignments.insurance_company_id', $this->insurance);
        if ($this->adjuster)
            $query->where('assignments.adjuster_id', $this->adjuster);

        // Apply unit type and global status filters
        if(!empty($this->unit_type))
        {
            $unit_type_ids = explode(',', $this->unit_type);
            $query->whereIn('assignments.unit_type_id', $unit_type_ids);
        }

        if(!empty($this->global_status))
        {
            $global_status_ids = explode(',', $this->global_status);
            $query->whereIn('assignments.status_id', $global_status_ids);
        }

        if (!empty($this->specificDate)) {
            $specificDate = Carbon::parse($this->specificDate)->startOfDay();
            $query->whereDate('assignment_invoices.gen_date', $specificDate);
        }

        if ($this->status === 'paid') {
            $query->where('assignment_invoices.mark_paid', 1);
        } elseif ($this->status === 'unpaid') {
            $query->whereRaw('ROUND((invoice_amount + (invoice_amount * COALESCE(taxpercentage, 0) / 100)), 2) > ROUND(COALESCE(amount_paid, 0), 2)');
        }

        if (!empty($this->searchbyinvoiceno)) {
            // Explode by comma, trim whitespace, and remove empty values
            $invoiceNumbers = collect(explode(',', $this->searchbyinvoiceno))
                ->map(fn($item) => preg_replace('/\s+/', '', $item))
                ->filter()
                ->unique()
                ->toArray();

            if (count($invoiceNumbers) > 0) {
                $query->where(function ($q) use ($invoiceNumbers) {
                    foreach ($invoiceNumbers as $invoiceNo) {
                        $q->orWhere('assignment_invoices.invoiceno', $invoiceNo);
                    }
                });
            }
        }

        if (!empty($this->searchbyfileno)) {
            $keywords = collect(explode(',', $this->searchbyfileno))
                ->map(fn($item) => trim(preg_replace('/\s+/', '', $item)))
                ->filter()
                ->unique()
                ->toArray();

            if (count($keywords) > 0) {
                $query->where(function ($q) use ($keywords) {
                    foreach ($keywords as $keyword) {
                        $q->orWhere(function ($subQuery) use ($keyword) {
                            $subQuery->where('file_no', 'like', "%$keyword%")
                                ->orWhere('claim_number', 'like', "%$keyword%")
                                ->orWhere('policy_number', 'like', "%$keyword%")
                                ->orWhere('business_name', 'like', "%$keyword%");
                        });
                    }
                });
            }
        }

        if (!empty($this->fromDate) && !empty($this->toDate)) {
            $from = Carbon::parse($this->fromDate)->startOfDay();
            $to = Carbon::parse($this->toDate)->endOfDay();

            $query->whereBetween('assignment_invoices.gen_date', [$from, $to]);
        } elseif (!empty($this->termname)) {
            switch ($this->termname) {
                case '30':
                    $start = now()->subDays(30)->startOfDay();
                    $end = now()->endOfDay();
                    $query->whereBetween('assignment_invoices.gen_date', [$start, $end]);
                    break;

                case '31':
                    $end = now()->subDays(31)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;

                case '60':
                    $start = now()->subDays(60)->startOfDay();
                    $end = now()->subDays(30)->endOfDay();
                    $query->whereBetween('assignment_invoices.gen_date', [$start, $end]);
                    break;

                case '61':
                    $end = now()->subDays(61)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;

                case '90':
                    $start = now()->subDays(90)->startOfDay();
                    $end = now()->subDays(60)->endOfDay();
                    $query->whereBetween('assignment_invoices.gen_date', [$start, $end]);
                    break;

                case '91':
                    $end = now()->subDays(91)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;
            }
        }

        // Get all records without pagination
        $records = $query->orderByDesc('assignment_invoices.gen_date')->get();

        // Calculate totals
        $totalinvoiceAmount = $records->sum('invoice_amount');
        $totalTaxAmount = $records->sum(function($record) {
            return ($record->invoice_amount * $record->taxpercentage) / 100;
        });
        $totalpaidAmount = $records->sum('amount_paid');

        // Generate CSV content
        $filename = 'receivable_report_' . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($records, $totalinvoiceAmount, $totalTaxAmount, $totalpaidAmount) {
            $file = fopen('php://output', 'w');
            
            // Add CSV headers
            fputcsv($file, [
                'File#',
                'Invoice #',
                'Owner Name',
                'Office',
                'Insurance',
                'Bulk Paying?',
                'Claim #',
                'Policy #',
                'Invoice Date',
                'Total',
                'Tax',
                'Amount Paid',
                'Balance'
            ]);

            // Add data rows
            foreach ($records as $record) {
                $taxAmount = (!empty($record->finaltotal) && !empty($record->taxpercentage))
                    ? ($record->finaltotal * $record->taxpercentage) / 100
                    : 0;
                
                $balance = $record->finaltotal + $taxAmount - $record->amount_paid;
                $ownerName = trim($record->business_first_name . ' ' . $record->business_last_name);
                
                fputcsv($file, [
                    $record->file_no,
                    $record->invoiceno,
                    $ownerName,
                    $record->officename,
                    $record->insurancecompanyname,
                    $record->paying_type == 'single' ? 'NO' : 'YES',
                    $record->claim_number,
                    $record->policy_number,
                    Carbon::parse($record->gen_date)->format('m-d-Y'),
                    number_format($record->finaltotal, 2),
                    number_format($taxAmount, 2),
                    number_format($record->amount_paid, 2),
                    number_format($balance, 2)
                ]);
            }

            // Add totals row
            fputcsv($file, [
                'TOTALS',
                '',
                '',
                '',
                '',
                '',
                '',
                '',
                '',
                number_format($totalinvoiceAmount, 2),
                number_format($totalTaxAmount, 2),
                number_format($totalpaidAmount, 2),
                number_format(($totalinvoiceAmount + $totalTaxAmount) - $totalpaidAmount, 2)
            ]);

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function calculateBulkPaymentAmount()
    {
        $total_balance = 0;
        foreach ($this->invoices_selected as $invoiceID) {

            \Log::info('Processing invoice v2: ' . $invoiceID);

            $invoice = DB::table('assignment_invoices')->where('id', $invoiceID)->first();

            $invoice_amount = $invoice->invoice_amount ?? 0;
            $amount_paid = $invoice->amount_paid ?? 0;
            $taxpercentage = $invoice->taxpercentage ?? 0;
            $totaldue = $invoice_amount + ($invoice_amount * $taxpercentage / 100);
            $balance = $totaldue - $amount_paid;

            \Log::info('Balance: ' . $balance);
            \Log::info('Total balance: ' . $total_balance);

            $total_balance += $balance;
        }

        $this->bulk_payment_amount = number_format($total_balance, 2);
    }


    public function buildInvoiceQuery()
{
    // Build the same query as in render() method but without pagination
        $query = DB::table('assignment_invoices')
            ->leftJoin('assignments', 'assignments.id', '=', 'assignment_invoices.assignment_id')
            ->leftJoin('insurance_company', 'assignments.insurance_company_id', '=', 'insurance_company.id')
            ->leftJoin('franchise', 'assignments.franchise_id', '=', 'franchise.id')
            ->select(
                'assignments.*',
                'assignment_invoices.id as invoiceID',
                'franchise.company_name as officename',
                'insurance_company.company_name as insurancecompanyname',
                'assignment_invoices.invoice_amount as finaltotal',
                'assignment_invoices.invoiceno',
                'assignment_invoices.gen_date',
                'assignment_invoices.invoice_amount',
                'assignment_invoices.taxpercentage',
                'assignment_invoices.amount_paid',
                 'assignment_invoices.invoice_status',
                'insurance_company.paying_type'
            );

        $query->where('assignment_invoices.invoice_amount', '>', 0);
        $query->whereNotNull('assignment_invoices.gen_date');
        $query->where('assignment_invoices.invoice_status', '!=', 'Void');

        // Apply filters
        if ($this->franchise)
            $query->where('assignments.franchise_id', $this->franchise);
        if ($this->insurance)
            $query->where('assignments.insurance_company_id', $this->insurance);
        if ($this->adjuster)
            $query->where('assignments.adjuster_id', $this->adjuster);

        // Apply unit type and global status filters
        if(!empty($this->unit_type))
        {
            $unit_type_ids = explode(',', $this->unit_type);
            $query->whereIn('assignments.unit_type_id', $unit_type_ids);
        }

        if(!empty($this->global_status))
        {
            $global_status_ids = explode(',', $this->global_status);
            $query->whereIn('assignments.status_id', $global_status_ids);
        }

        if (!empty($this->specificDate)) {
            $specificDate = Carbon::parse($this->specificDate)->startOfDay();
            $query->whereDate('assignment_invoices.gen_date', $specificDate);
        }

        if ($this->status === 'paid') {
            $query->where('assignment_invoices.mark_paid', 1);
        } elseif ($this->status === 'unpaid') {
            $query->whereRaw('ROUND((invoice_amount + (invoice_amount * COALESCE(taxpercentage, 0) / 100)), 2) > ROUND(COALESCE(amount_paid, 0), 2)');
        }

        if (!empty($this->searchbyinvoiceno)) {
            // Explode by comma, trim whitespace, and remove empty values
            $invoiceNumbers = collect(explode(',', $this->searchbyinvoiceno))
                ->map(fn($item) => preg_replace('/\s+/', '', $item))
                ->filter()
                ->unique()
                ->toArray();

            if (count($invoiceNumbers) > 0) {
                $query->where(function ($q) use ($invoiceNumbers) {
                    foreach ($invoiceNumbers as $invoiceNo) {
                        $q->orWhere('assignment_invoices.invoiceno', $invoiceNo);
                    }
                });
            }
        }

        if (!empty($this->searchbyfileno)) {
            $keywords = collect(explode(',', $this->searchbyfileno))
                ->map(fn($item) => trim(preg_replace('/\s+/', '', $item)))
                ->filter()
                ->unique()
                ->toArray();

            if (count($keywords) > 0) {
                $query->where(function ($q) use ($keywords) {
                    foreach ($keywords as $keyword) {
                        $q->orWhere(function ($subQuery) use ($keyword) {
                            $subQuery->where('file_no', 'like', "%$keyword%")
                                ->orWhere('claim_number', 'like', "%$keyword%")
                                ->orWhere('policy_number', 'like', "%$keyword%")
                                ->orWhere('business_name', 'like', "%$keyword%");
                        });
                    }
                });
            }
        }

        if (!empty($this->fromDate) && !empty($this->toDate)) {
            $from = Carbon::parse($this->fromDate)->startOfDay();
            $to = Carbon::parse($this->toDate)->endOfDay();

            $query->whereBetween('assignment_invoices.gen_date', [$from, $to]);
        } elseif (!empty($this->termname)) {
            switch ($this->termname) {
                case '30':
                    $start = now()->subDays(30)->startOfDay();
                    $end = now()->endOfDay();
                    $query->whereBetween('assignment_invoices.gen_date', [$start, $end]);
                    break;

                case '31':
                    $end = now()->subDays(31)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;

                case '60':
                    $start = now()->subDays(60)->startOfDay();
                    $end = now()->subDays(30)->endOfDay();
                    $query->whereBetween('assignment_invoices.gen_date', [$start, $end]);
                    break;

                case '61':
                    $end = now()->subDays(61)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;

                case '90':
                    $start = now()->subDays(90)->startOfDay();
                    $end = now()->subDays(60)->endOfDay();
                    $query->whereBetween('assignment_invoices.gen_date', [$start, $end]);
                    break;

                case '91':
                    $end = now()->subDays(91)->endOfDay();
                    $query->where('assignment_invoices.gen_date', '<', $end);
                    break;
            }
        }

        // Get all records without pagination
        return $query->orderByDesc('assignment_invoices.gen_date');

}


public function exportFilteredInvoicesPdf()
    {
    
    ini_set('memory_limit', '2G');
    set_time_limit(600);

    $query = $this->buildInvoiceQuery();
    $invoices = $query->cursor(); // lazy collection

    $data = [
        'invoices' => $invoices,
        'generated_at' => now()->toDateTimeString(),
        // can't use sum() directly on cursor, so loop inside blade if needed
    ];

    $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.exportreceivablepdf', $data)
        ->setPaper('a4', 'landscape')
        ->setOptions([
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => true,
        ]);

    $filename = 'invoices_' . now()->format('Ymd_His') . '.pdf';

    return response()->streamDownload(function () use ($pdf) {
        echo $pdf->output();
    }, $filename);

    }
}

