<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use Livewire\WithPagination;

class DashboardStats extends Component
{
    use WithPagination;
    public $totalBillAmount = 0;
    public $paidBillAmount = 0;
    public $unpaidBillAmount = 0;
    public $chartDates = [];
    public $chartDataCount = [];
    public $piechartData = [];
    public $premiumcompanyData = [];
    public $longestcycletime = [];
    public $multiple_franchise = [];
    public $completed_status_id;
    public $closed_status_id;
    public function mount()
    {
        $tempRecord = GlobalStatus::whereRaw('LOWER(status) = ?', [strtolower('Completed')])->first();
        if ($tempRecord) {
            $this->completed_status_id = $tempRecord->id;
        }
        $tempRecord = GlobalStatus::whereRaw('LOWER(status) = ?', [strtolower('Closed')])->first();
        if ($tempRecord) {
            $this->closed_status_id = $tempRecord->id;
        }
        
        $this->loadStats();
    }

    public function loadStats()
    {
        // If user is office admin, then limit the records to the multiple franchise only;
        if (auth()->user()->franchise_id != 0 && auth()->user()->franchise_id != '' && auth()->user()->franchise_id != null)
        {
            $franchises = auth()->user()->multiple_franchise;
            $this->multiple_franchise = $franchises 
                ? array_map('trim', explode(',', $franchises)) 
                : [];
            $this->multiple_franchise[] = auth()->user()->franchise;
            $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
        }

        $chartDates = [];
        $chartDataCount = [];

        $startDate = now()->startOfMonth();
        $endDate = now()->endOfMonth();

        while ($startDate->lte($endDate)) {
            $label = $startDate->format('d-M');
            $chartDates[] = $label;

            $created = Assignment::where('created_date', $startDate->format('Y-m-d'));

            if (count($this->multiple_franchise) > 0) {
                $created = $created->whereIn('franchise_id', $this->multiple_franchise);
            }

            $created = $created->count();


            $chartDataCount[] = $created;
            $startDate->addDay();
        }

        // Assign to public properties
        $this->chartDates = $chartDates;
        $this->chartDataCount = $chartDataCount;
        $this->fetchSummary();
        $this->TopCustomersChart();
        $this->TopPremiumCustomers();
        $this->Top10longestcycletime();
    }

    public function fetchSummary()
    {
        $franchiseFilter = session('franchise_filter'); // Optional: from session or request

        $queryBase = DB::table('assignment_invoices')
            ->leftJoin('assignments', 'assignments.id', '=', 'assignment_invoices.assignment_id');

        if ($franchiseFilter) {
            $queryBase->whereIn('franchise_id', explode(',', $franchiseFilter));
        }

        $this->totalBillAmount = (clone $queryBase)
            ->where('invoice_status', '!=', 'Void')->whereNotNull('gen_date')
            ->sum(DB::raw('invoice_amount * (1 + taxpercentage / 100)'));


        


        $this->unpaidBillAmount = (clone $queryBase)
            ->where('invoice_status', '!=', 'Void')
            ->whereNotNull('gen_date')
            ->where('assignment_invoices.invoice_amount', '>', 0)
            ->sum(DB::raw('(invoice_amount * (1 + COALESCE(taxpercentage, 0) / 100)) - COALESCE(amount_paid, 0)'));

        $invoices = (clone $queryBase)
            ->where('invoice_status', '!=', 'Void')
            ->whereNotNull('gen_date')
            ->get(['invoice_amount', 'taxpercentage', 'amount_paid']);

        $unpaidTotal = 0;

        foreach ($invoices as $invoice) {
            $amount = (float) $invoice->invoice_amount;
            $tax = (float) $invoice->taxpercentage;
            $paid = (float) $invoice->amount_paid;

            $totalWithTax = $amount * (1 + $tax / 100);
            $balance = $totalWithTax - $paid;

            $unpaidTotal += $balance;
        }
        $this->unpaidphp = $unpaidTotal;

        $this->paidBillAmount = (clone $queryBase)
            //->where('invoice_status', '=', 'Paid')
            ->where('invoice_status', '!=', 'Void')
            ->where('assignment_invoices.invoice_amount', '>', 0)
            ->whereNotNull('gen_date')
            ->sum('amount_paid');
    }

    public function TopCustomersChart()
    {
        $query = DB::table('assignments');

        if (count($this->multiple_franchise) > 0) {
            $query = $query->whereIn('assignments.franchise_id', $this->multiple_franchise);
        }

        $this->piechartData =  $query->join('insurance_company', 'assignments.insurance_company_id', '=', 'insurance_company.id')
            ->join('assignment_invoices', 'assignments.id', '=', 'assignment_invoices.assignment_id')
            ->select(
                'insurance_company.company_name',
                DB::raw('COUNT(assignments.file_no) as total_assignments')
            )
            ->groupBy('insurance_company.company_name')
            ->orderByDesc('total_assignments')
            ->limit(10)
            ->get()
            ->toArray();

    }

    public function TopPremiumCustomers()
    {
        $query = DB::table('assignments');

        if (count($this->multiple_franchise) > 0) {
            $query = $query->whereIn('assignments.franchise_id', $this->multiple_franchise);
        }

        $this->premiumcompanyData = $query
            ->join('assignment_invoices', 'assignments.id', '=', 'assignment_invoices.assignment_id')
            ->join('insurance_company', 'assignments.insurance_company_id', '=', 'insurance_company.id')
            ->select(
                'insurance_company.company_name',
                DB::raw('COUNT(assignments.file_no) as total_assignments'),
                DB::raw('SUM(assignment_invoices.invoice_amount) as totalinvoiceamount')
            )
            ->groupBy('insurance_company.company_name')
            ->orderByDesc('totalinvoiceamount')
            ->limit(10)
            ->get()
            ->toArray();
    }


    public function Top10longestcycletime()
    {

        $query = DB::table('assignments');

        if (count($this->multiple_franchise) > 0) {
            $query = $query->whereIn('assignments.franchise_id', $this->multiple_franchise);
        }

        $this->longestcycletime = $query
            ->join('insurance_company', 'assignments.insurance_company_id', '=', 'insurance_company.id')
            ->join('franchise', 'assignments.franchise_id', '=', 'franchise.id')
            ->select('*', 'franchise.company_name as officename', 'insurance_company.company_name as insurancename', 'assignments.created_date as assignmentopendate', DB::raw('DATEDIFF(assignments.complete_date, assignments.created_date) as day_diff'))
            ->whereIn('assignments.status_id', [$this->completed_status_id, $this->closed_status_id])
            ->where('assignments.complete_date', '!=', '')
            ->whereNotNull('assignments.complete_date')
            ->orderByDesc('day_diff')
            ->limit(20)
            ->get()
            ->toArray();


    }

    public function render()
    {
        return view('livewire.dashboard-stats');
    }
}

