<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use App\Models\UnitType;
use Livewire\WithPagination;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ClientKpi extends Component
{
    use WithPagination;
    public $allFranchises = [];
    public $allInsurances = [];
    public $allAdjusters = [];
    public $allAppraisers = [];
    public $perPage = 25;
    public $insurance;
    public $franchise;
    public $adjuster;
    public $appraiser;
    public $from_date;
    public $to_date;
    public $exclude_weekends = false;
    protected $paginationTheme = 'bootstrap';
    public $unit_types = [];
    public $unit_type = '';
    public $global_statuses = [];
    public $global_status = '';
    public $multiple_franchise = [];

    public function search()
    {
        $this->resetPage();
        $this->refreshOptions();
    }
    
    public function mount()
    {
        // If user is office admin, then limit the records to the multiple franchise only;
        if (auth()->user()->franchise != 0 && auth()->user()->franchise != '' && auth()->user()->franchise != null)
        {
            $franchises = auth()->user()->multiple_franchise;
            $this->multiple_franchise = $franchises 
                ? array_map('trim', explode(',', $franchises)) 
                : [];
            $this->multiple_franchise[] = auth()->user()->franchise;
            $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
        }

        $this->refreshOptions();
    }


    public function exportCsv()
    {
        $data = $this->getQuery()->get(); // extract common query to reuse

        return response()->streamDownload(function () use ($data) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['Insurance', 'Assignment Count', 'Avg Days', 'Supp Ratio %', 'Supp Assignment Count', 'Supp Avg Days']);

            foreach ($data as $row) {
                fputcsv($handle, [
                    $row->company_name,
                    $row->assignment_count,
                    $row->avg_days,
                    $row->supp_ratio,
                    $row->supp_count,
                    $row->supp_avg_days
                ]);
            }

            fclose($handle);
        }, 'insurance_stats_' . now()->format('Ymd') . '.csv');
    }

    public function getQuery()
    {
        if ($this->exclude_weekends) {
            $query = DB::table('assignments as a')
                ->join('insurance_company as i', 'a.insurance_company_id', '=', 'i.id')
                ->select(
                    'i.company_name',
                    DB::raw('COUNT(a.file_no) as assignment_count'),
                    DB::raw('COALESCE(ROUND(AVG(
                        ABS(TIMESTAMPDIFF(DAY, a.created_date, a.complete_date)) - 
                        (FLOOR(ABS(TIMESTAMPDIFF(DAY, a.created_date, a.complete_date)) / 7) * 2) - 
                        CASE 
                            WHEN WEEKDAY(a.created_date) <= 4 AND WEEKDAY(a.complete_date) >= 5 THEN 
                                GREATEST(0, 5 - WEEKDAY(a.created_date)) + GREATEST(0, WEEKDAY(a.complete_date) - 4)
                            WHEN WEEKDAY(a.created_date) > WEEKDAY(a.complete_date) THEN 
                                GREATEST(0, 5 - WEEKDAY(a.created_date)) + GREATEST(0, WEEKDAY(a.complete_date) - 4)
                            ELSE 0
                        END
                    ), 2), 0) as avg_days'),
                    DB::raw('SUM(CASE WHEN a.supplement = "YES" THEN 1 ELSE 0 END) as supp_count'),
                    DB::raw('COALESCE(ROUND(SUM(CASE WHEN a.supplement = "YES" THEN 1 ELSE 0 END) / COUNT(a.file_no) * 100, 2), 0) as supp_ratio'),
                    DB::raw('COALESCE(ROUND(AVG(CASE WHEN a.supplement = "YES" THEN 
                        ABS(TIMESTAMPDIFF(DAY, a.created_date, a.complete_date)) - 
                        (FLOOR(ABS(TIMESTAMPDIFF(DAY, a.created_date, a.complete_date)) / 7) * 2) - 
                        CASE 
                            WHEN WEEKDAY(a.created_date) <= 4 AND WEEKDAY(a.complete_date) >= 5 THEN 
                                GREATEST(0, 5 - WEEKDAY(a.created_date)) + GREATEST(0, WEEKDAY(a.complete_date) - 4)
                            WHEN WEEKDAY(a.created_date) > WEEKDAY(a.complete_date) THEN 
                                GREATEST(0, 5 - WEEKDAY(a.created_date)) + GREATEST(0, WEEKDAY(a.complete_date) - 4)
                            ELSE 0
                        END
                    ELSE NULL END), 2), 0) as supp_avg_days')
                )
                ->when($this->from_date, function ($query) {
                    $query->where('a.created_date', '>=', $this->from_date . ' 00:00:00');
                })
                ->when($this->to_date, function ($query) {
                    $query->where('a.created_date', '<=', $this->to_date . ' 23:59:59');
                })
                ->when($this->insurance, function ($query) {
                    $ids = explode(',', $this->insurance);
                    $query->whereIn('i.id', $ids);
                })
                ->when($this->franchise, function ($query) {
                    $ids = explode(',', $this->franchise);
                    $query->whereIn('a.franchise_id', $ids);
                })
                ->when($this->adjuster, function ($query) {
                    $ids = explode(',', $this->adjuster);
                    $query->whereIn('a.adjuster_id', $ids);
                })
                ->when($this->appraiser, function ($query) {
                    $ids = explode(',', $this->appraiser);
                    $query->whereIn('a.appraiser_id', $ids);
                })
                ->when($this->unit_type, function ($query) {
                    $ids = explode(',', $this->unit_type);
                    $query->whereIn('a.unit_type_id', $ids);
                })
                ->when($this->global_status, function ($query) {
                    $ids = explode(',', $this->global_status);
                    $query->whereIn('a.status_id', $ids);
                });

                if (count($this->multiple_franchise) > 0) {
                    $query = $query->whereIn('a.franchise_id', $this->multiple_franchise);
                }

                $query = $query->groupBy('i.id')
                ->orderBy('assignment_count', 'desc');
        } else {
            $query = DB::table('assignments as a')
                ->join('insurance_company as i', 'a.insurance_company_id', '=', 'i.id')
                ->select(
                    'i.company_name',
                    DB::raw('COUNT(a.file_no) as assignment_count'),
                    DB::raw('COALESCE(ROUND(AVG(ABS(TIMESTAMPDIFF(DAY, a.created_date, a.complete_date))), 2), 0) as avg_days'),
                    DB::raw('SUM(CASE WHEN a.supplement = "YES" THEN 1 ELSE 0 END) as supp_count'),
                    DB::raw('COALESCE(ROUND(SUM(CASE WHEN a.supplement = "YES" THEN 1 ELSE 0 END) / COUNT(a.file_no) * 100, 2), 0) as supp_ratio'),
                    DB::raw('COALESCE(ROUND(AVG(CASE WHEN a.supplement = "YES" THEN ABS(TIMESTAMPDIFF(DAY, a.created_date, a.complete_date)) ELSE NULL END), 2), 0) as supp_avg_days')
                )
                ->when($this->from_date, function ($query) {
                    $query->where('a.created_date', '>=', $this->from_date . ' 00:00:00');
                })
                ->when($this->to_date, function ($query) {
                    $query->where('a.created_date', '<=', $this->to_date . ' 23:59:59');
                })
                ->when($this->insurance, function ($query) {
                    $ids = explode(',', $this->insurance);
                    $query->whereIn('i.id', $ids);
                })
                ->when($this->franchise, function ($query) {
                    $ids = explode(',', $this->franchise);
                    $query->whereIn('a.franchise_id', $ids);
                })
                ->when($this->adjuster, function ($query) {
                    $ids = explode(',', $this->adjuster);
                    $query->whereIn('a.adjuster_id', $ids);
                })
                ->when($this->appraiser, function ($query) {
                    $ids = explode(',', $this->appraiser);
                    $query->whereIn('a.appraiser_id', $ids);
                })
                ->when($this->unit_type, function ($query) {
                    $ids = explode(',', $this->unit_type);
                    $query->whereIn('a.unit_type_id', $ids);
                })
                ->when($this->global_status, function ($query) {
                    $ids = explode(',', $this->global_status);
                    $query->whereIn('a.status_id', $ids);
                });

                if (count($this->multiple_franchise) > 0) {
                    $query = $query->whereIn('a.franchise_id', $this->multiple_franchise);
                }

                $query = $query->groupBy('i.id')
                ->orderBy('assignment_count', 'desc');
        }

        return $query;
    }

    public function render()
    {

        // $allInsurances = DB::table('insurance_company')->orderBy('company_name', 'asc')->get();
        // $allFranchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->get();
        $getQuery = $this->getQuery()->paginate($this->perPage);

        return view('livewire.client-kpi', [
            'stats' => $getQuery,
            // 'allInsurances' => $allInsurances,
            // 'allFranchises' => $allFranchises,
        ]);


    }

    public function refreshOptions()
    {
        // Refresh the options data
        $this->allInsurances = DB::table('insurance_company')->orderBy('company_name', 'asc')->get();
        $this->allAdjusters = DB::select(" SELECT * FROM users join insurance_company on insurance_company.id=users.insurance_company_id AND users.role=4   order by users.name asc ");
        if (count($this->multiple_franchise) > 0) {
            $this->allFranchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->whereIn('id', $this->multiple_franchise)->get();
        } else {
            $this->allFranchises = DB::table('franchise')->orderBy('franchisecode', 'asc')->get();
        }
        $this->allAppraisers = DB::select(" SELECT * FROM users WHERE users.role=2   order by users.name asc ");
        $this->unit_types = UnitType::all();
        $this->global_statuses = GlobalStatus::all();

    }
}

