<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use App\Models\InsuranceCompany;
use Livewire\WithPagination;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Pagination\LengthAwarePaginator;

class ClientHealth extends Component
{
    
    public $insuranceCompanies = [];
    public $monthlyData = [];
    public $selectedYear;
    public $trendData = [];
    public $errorMessage = '';
    public $searchTerm = '';
    public $filterTrend = '';
    public $perPage = 25;
    public $currentPage = 1;
    public $sortBy = 'company_name';
    public $sortDirection = 'asc';
    public $isLoading = false;
    public $multiple_franchise = [];
    public $supplement_status_id;

    public function mount()
    {
        try {

            $tempRecord = GlobalStatus::whereRaw('LOWER(status) = ?', [strtolower('Supplement')])->first();
            if ($tempRecord) {
                $this->supplement_status_id = $tempRecord->id;
            }

            // If user is office admin, then limit the records to the multiple franchise only;
            if (auth()->user()->franchise != 0 && auth()->user()->franchise != '' && auth()->user()->franchise != null)
            {
                $franchises = auth()->user()->multiple_franchise;
                $this->multiple_franchise = $franchises 
                    ? array_map('trim', explode(',', $franchises)) 
                    : [];
                $this->multiple_franchise[] = auth()->user()->franchise;
                $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
            }

            $this->selectedYear = date('Y');
            $this->loadData();
        } catch (\Exception $e) {
            Log::error('ClientHealth mount error: ' . $e->getMessage());
            $this->errorMessage = 'Error loading data: ' . $e->getMessage();
        }
    }

    public function loadData()
    {
        try {
            // Get all insurance companies
            $this->insuranceCompanies = InsuranceCompany::all();
            
            // Calculate monthly assignment counts for the selected year
            $this->calculateMonthlyData();
            
            // Calculate trends (increasing/decreasing)
            $this->calculateTrends();
        } catch (\Exception $e) {
            Log::error('ClientHealth loadData error: ' . $e->getMessage());
            $this->errorMessage = 'Error loading data: ' . $e->getMessage();
        }
    }

    public function calculateMonthlyData()
    {
        $this->monthlyData = [];
        
        foreach ($this->insuranceCompanies as $company) {
            $monthlyCounts = [];
            
            for ($month = 1; $month <= 12; $month++) {
                try {
                    $_query = Assignment::where('insurance_company_id', $company->id)
                        ->whereYear('created_date', $this->selectedYear)
                        ->whereMonth('created_date', $month)
                        ->where('status_id', '!=', $this->supplement_status_id);
                    if (count($this->multiple_franchise) > 0) {
                        $_query = $_query->whereIn('franchise_id', $this->multiple_franchise);
                    }
                    $count = $_query->count();
                    
                    $monthlyCounts[$month] = $count;
                } catch (\Exception $e) {
                    Log::error("Error counting assignments for company {$company->id}, month {$month}: " . $e->getMessage());
                    $monthlyCounts[$month] = 0;
                }
            }
            
            $this->monthlyData[$company->id] = $monthlyCounts;
        }
    }

    public function calculateTrends()
    {
        $this->trendData = [];
        
        // Get current date information
        $currentDate = Carbon::now();
        $currentYear = $currentDate->year;
        $currentMonth = $currentDate->month;
        $daysInCurrentMonth = $currentDate->daysInMonth;
        $daysPassedThisMonth = $currentDate->day;
        
        // Calculate the proportion of the month that has passed
        $monthProgress = $daysPassedThisMonth / $daysInCurrentMonth;
        
        foreach ($this->insuranceCompanies as $company) {
            $monthlyCounts = $this->monthlyData[$company->id];
            
            // Get current month data
            $currentMonthCount = $monthlyCounts[$currentMonth] ?? 0;
            
            // Get previous month data
            $previousMonth = $currentMonth - 1;
            $previousYear = $currentYear;
            if ($previousMonth < 1) {
                $previousMonth = 12;
                $previousYear = $currentYear - 1;
            }
            
            // Get previous month's assignment count for the same period
            $previousMonthCount = 0;
            if ($previousYear == $this->selectedYear) {
                $previousMonthCount = $monthlyCounts[$previousMonth] ?? 0;
            } else {
                // If previous month is in different year, we need to calculate it
                $_query = Assignment::where('insurance_company_id', $company->id)
                    ->whereYear('created_date', $previousYear)
                    ->whereMonth('created_date', $previousMonth);
                if (count($this->multiple_franchise) > 0) {
                    $_query = $_query->whereIn('franchise_id', $this->multiple_franchise);
                }
                $previousMonthCount = $_query->count();
            }
            
            // Calculate trend by comparing current month progress with previous month
            $trend = 'stable';
            $trendPercentage = 0;
            
            if ($previousMonthCount > 0) {
                // Project current month's total based on progress
                $projectedCurrentMonthTotal = $currentMonthCount / $monthProgress;
                
                // Calculate percentage change
                $trendPercentage = (($projectedCurrentMonthTotal - $previousMonthCount) / $previousMonthCount) * 100;
                
                if ($trendPercentage >= 1) {
                    $trend = 'increasing';
                } elseif ($trendPercentage < -1) {
                    $trend = 'decreasing';
                }
            } elseif ($currentMonthCount > 0) {
                // If no previous month data but current month has data, mark as increasing
                $trend = 'increasing';
                $trendPercentage = 100; // 100% increase from 0
            }
            
            $this->trendData[$company->id] = [
                'trend' => $trend,
                'percentage' => round($trendPercentage, 1),
                'total_assignments' => array_sum($monthlyCounts),
                'current_month_progress' => round($monthProgress * 100, 1),
                'days_passed' => $daysPassedThisMonth,
                'days_in_month' => $daysInCurrentMonth
            ];
        }
    }

    public function updatedSelectedYear()
    {
        $this->loadData();
    }

    public function getFilteredCompaniesProperty()
    {
        $companies = $this->insuranceCompanies;
        
        // Filter by search term
        if (!empty($this->searchTerm)) {
            $companies = $companies->filter(function($company) {
                return stripos($company->company_name ?? '', $this->searchTerm) !== false;
            });
        }
        
        // Filter by trend
        if (!empty($this->filterTrend)) {
            $companies = $companies->filter(function($company) {
                $trendData = $this->trendData[$company->id] ?? [];
                return ($trendData['trend'] ?? '') === $this->filterTrend;
            });
        }
        
        // Sort companies based on the sort field
        if ($this->sortBy === 'total_assignments') {
            // Sort by total assignments (computed field)
            $companies = $companies->sortBy(function($company) {
                $trendData = $this->trendData[$company->id] ?? [];
                return $trendData['total_assignments'] ?? 0;
            }, SORT_NUMERIC, $this->sortDirection === 'desc');
        } else {
            // Sort by company properties
            $companies = $companies->sortBy([
                [$this->sortBy, $this->sortDirection]
            ]);
        }
        
        return $companies;
    }

    public function getPaginatedCompaniesProperty()
    {
        $filteredCompanies = $this->filteredCompanies;
        $total = $filteredCompanies->count();
        
        // Reset to first page if current page is beyond total pages
        $totalPages = ceil($total / $this->perPage);
        if ($this->currentPage > $totalPages && $totalPages > 0) {
            $this->currentPage = 1;
        }
        
        return $filteredCompanies->forPage($this->currentPage, $this->perPage);
    }

    public function getTotalPagesProperty()
    {
        return ceil($this->filteredCompanies->count() / $this->perPage);
    }

    public function goToPage($page)
    {
        $this->isLoading = true;
        $this->currentPage = $page;
        $this->isLoading = false;
    }

    public function nextPage()
    {
        if ($this->currentPage < $this->totalPages) {
            $this->isLoading = true;
            $this->currentPage++;
            $this->isLoading = false;
        }
    }

    public function previousPage()
    {
        if ($this->currentPage > 1) {
            $this->isLoading = true;
            $this->currentPage--;
            $this->isLoading = false;
        }
    }

    public function updatedSearchTerm()
    {
        $this->currentPage = 1; // Reset to first page when searching
    }

    public function updatedFilterTrend()
    {
        $this->currentPage = 1; // Reset to first page when filtering
    }

    public function updatedPerPage()
    {
        $this->currentPage = 1; // Reset to first page when changing page size
    }

    public function sortByField($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
        $this->currentPage = 1; // Reset to first page when sorting
    }

    public function exportToCsv()
    {
        try {
            $filename = 'customer_health_report_' . $this->selectedYear . '_' . date('Y-m-d') . '.csv';
            
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            ];
            
            $callback = function() {
                $file = fopen('php://output', 'w');
                
                // CSV Headers
                $headers = [
                    'Insurance Company',
                    'Total Assignments',
                    'Trend',
                    'Change %',
                    'Month Progress %',
                    'Days Passed',
                    'Days in Month',
                    'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
                    'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'
                ];
                fputcsv($file, $headers);
                
                // Data rows
                foreach ($this->insuranceCompanies as $company) {
                    $trendData = $this->trendData[$company->id] ?? [];
                    $monthlyData = $this->monthlyData[$company->id] ?? [];
                    
                    $row = [
                        $company->company_name ?? 'Unknown Company',
                        $trendData['total_assignments'] ?? 0,
                        $trendData['trend'] ?? 'N/A',
                        ($trendData['percentage'] ?? 0) . '%',
                        ($trendData['current_month_progress'] ?? 0) . '%',
                        $trendData['days_passed'] ?? 0,
                        $trendData['days_in_month'] ?? 0,
                    ];
                    
                    // Add monthly data
                    for ($month = 1; $month <= 12; $month++) {
                        $row[] = $monthlyData[$month] ?? 0;
                    }
                    
                    fputcsv($file, $row);
                }
                
                fclose($file);
            };
            
            return response()->stream($callback, 200, $headers);
        } catch (\Exception $e) {
            Log::error('ClientHealth export error: ' . $e->getMessage());
            $this->errorMessage = 'Error exporting data: ' . $e->getMessage();
        }
    }

    function render()
    {
        return view('livewire.client-health');
    }
}

