<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\DB;
use App\Models\Assignment;
use App\Models\GlobalStatus;
use App\Models\UnitType;
use Livewire\WithPagination;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Pagination\LengthAwarePaginator;

class AppraiserReport extends Component
{
	
    use WithPagination;
    protected $paginationTheme = 'bootstrap';

    public $exclude_supplement = false;
    public $from_date = '';
    public $to_date = '';
    public $franchise = '';
    public $insurance_company = '';
    public $exclude_weekend = false;
    public $isLoading = false;
    public $unit_types = [];
    public $unit_type = '';
    public $global_statuses = [];
    public $global_status = '';
    public $multiple_franchise = [];
	public $completed_status_id;
    public $closed_status_id;
	public $cancelled_status_id;

    public function formatCycleTime($hours)
    {
        $hours = abs($hours);
        $days = floor($hours / 24);
        $remainingHours = floor($hours % 24);
        $minutes = floor(($hours - floor($hours)) * 60);
        return "{$days}d {$remainingHours}h {$minutes}m";
    }

    public function mount()
    {
        // If user is office admin, then limit the records to the multiple franchise only;
        if (auth()->user()->franchise != 0 && auth()->user()->franchise != '' && auth()->user()->franchise != null)
        {
            $franchises = auth()->user()->multiple_franchise;
            $this->multiple_franchise = $franchises 
                ? array_map('trim', explode(',', $franchises)) 
                : [];
            $this->multiple_franchise[] = auth()->user()->franchise;
            $this->multiple_franchise = array_unique(array_filter($this->multiple_franchise));
        }

		$tempRecord = GlobalStatus::whereRaw('LOWER(status) = ?', [strtolower('Completed')])->first();
        if ($tempRecord) {
            $this->completed_status_id = $tempRecord->id;
        }

        $tempRecord = GlobalStatus::whereRaw('LOWER(status) = ?', [strtolower('Closed')])->first();
        if ($tempRecord) {
            $this->closed_status_id = $tempRecord->id;
        }

		$tempRecord = GlobalStatus::whereRaw('LOWER(status) = ?', [strtolower('Cancelled')])->first();
        if ($tempRecord) {
            $this->cancelled_status_id = $tempRecord->id;
        }

        $this->unit_types = UnitType::all();
        $this->global_statuses = GlobalStatus::all();
        $this->isLoading = true;
    }

    public function updating($property, $value)
    {
        \Log::info('updating property: '.$property);
        $this->isLoading = true;
    }

    public function updated($property, $value)
    {
        \Log::info('updated property: '.$property);

        // Set loading to true for any filter change
        $this->isLoading = true;

        $fields = ['exclude_supplement', 'exclude_weekend', 'from_date', 'to_date', 'franchise', 'insurance_company', 'unit_type', 'global_status'];

        if (in_array($property, $fields)) {
            $this->resetPage();
        }

    }

    public function render()
    {
        // Set loading to false when render is called (data is ready)
        $this->isLoading = false;

        $appraiserData = $this->getAppraiserData();
        // Calculate averages from all records, not just current page
        $allAppraisers = $this->getAllAppraiserData();
        $averages = $this->calculateAverages($allAppraisers);
        $totals = $this->calculateTotals($allAppraisers);

        return view('livewire.appraiser-report', [
            'appraiser_data' => $appraiserData, 
            'franchises' => $this->getFranchises(),
            'insurance_companies' => $this->getInsuranceCompanies(),
            'averages' => $averages,
            'totals' => $totals
        ]);
    }

    private function getAppraiserData()
    {
        $assignments = DB::table('users')
            ->join('assignments', 'users.id', '=', 'assignments.appraiser_id')
            ->where('users.role', 2)
            ->select(
                'users.id as user_id',
                'users.name',
                'assignments.created_date',
                'assignments.complete_date',
                'assignments.status_id'
            )
            ->when($this->exclude_supplement, function($query) {
                return $query->where(function ($q) {
                    $q->where('assignments.supplement', '!=', 'YES')
                        ->orWhereNull('assignments.supplement');
                });
            })
            ->when($this->from_date && $this->to_date, function($query) {
                return $query->whereBetween('assignments.created_date', [$this->from_date, $this->to_date]);
            })
            ->when($this->from_date && !$this->to_date, function($query) {
                return $query->where('assignments.created_date', '>=', $this->from_date);
            })
            ->when($this->to_date && !$this->from_date, function($query) {
                return $query->where('assignments.created_date', '<=', $this->to_date);
            })
            ->when($this->franchise, function($query) {
                $ids = explode(',', $this->franchise);
                return $query->whereIn('assignments.franchise_id', $ids);
            })
            ->when($this->insurance_company, function($query) {
                $ids = explode(',', $this->insurance_company);
                return $query->whereIn('assignments.insurance_company_id', $ids);
                // return $query->where('assignments.insurancecompanyid', $this->insurance_company);
            })
            ->when($this->unit_type, function($query) {
                $ids = explode(',', $this->unit_type);
                return $query->whereIn('assignments.unit_type_id', $ids);
            })
            ->when($this->global_status, function($query) {
                $ids = explode(',', $this->global_status);
                return $query->whereIn('assignments.status_id', $ids);
            });

            if (count($this->multiple_franchise) > 0) {
                $assignments->whereIn('assignments.franchise_id', $this->multiple_franchise);
            }

            $assignments = $assignments->get();

        // Now process results in PHP
        return $this->calculateAppraiserStats($assignments);
    }

    private function getAllAppraiserData()
    {
        $assignments = DB::table('users')
            ->join('assignments', 'users.id', '=', 'assignments.appraiser_id')
            ->where('users.role', 2)
            ->select(
                'users.id as user_id',
                'users.name',
                'assignments.created_date',
                'assignments.complete_date',
                'assignments.status_id'
            )
            ->when($this->exclude_supplement, function($query) {
                return $query->where(function ($q) {
                    $q->where('assignments.supplement', '!=', 'YES')
                        ->orWhereNull('assignments.supplement');
                });
            })
            ->when($this->from_date && $this->to_date, function($query) {
                return $query->whereBetween('assignments.created_date', [$this->from_date, $this->to_date]);
            })
            ->when($this->from_date && !$this->to_date, function($query) {
                return $query->where('assignments.created_date', '>=', $this->from_date);
            })
            ->when($this->to_date && !$this->from_date, function($query) {
                return $query->where('assignments.created_date', '<=', $this->to_date);
            })
            ->when($this->franchise, function($query) {
                $ids = explode(',', $this->franchise);
                return $query->whereIn('assignments.franchise_id', $ids);
            })
            ->when($this->insurance_company, function($query) {
                $ids = explode(',', $this->insurance_company);
                return $query->whereIn('assignments.insurance_company_id', $ids);
                // return $query->where('assignments.insurancecompanyid', $this->insurance_company);
            })
            ->when($this->unit_type, function($query) {
                $ids = explode(',', $this->unit_type);
                return $query->whereIn('assignments.unit_type_id', $ids);
            })
            ->when($this->global_status, function($query) {
                $ids = explode(',', $this->global_status);
                return $query->whereIn('assignments.status_id', $ids);
            });

            if (count($this->multiple_franchise) > 0) {
                $assignments->whereIn('assignments.franchise_id', $this->multiple_franchise);
            }

            $assignments = $assignments->get();

        // Now process results in PHP without pagination
        return $this->calculateAppraiserStatsRaw($assignments);
    }

    private function calculateBusinessHoursDiff($start, $end)
    {
        $startDate = new DateTime($start);
        $endDate = new DateTime($end);

        if ($endDate < $startDate) {
            return 0;
        }

        $totalSeconds = $endDate->getTimestamp() - $startDate->getTimestamp();

        // Iterate day by day to subtract weekend seconds
        $current = clone $startDate;
        $weekendSeconds = 0;

        while ($current < $endDate) {
            $dayOfWeek = (int) $current->format('N'); // 1=Mon ... 7=Sun

            // Start and end of this day in timestamps
            $dayStart = (clone $current)->setTime(0,0,0);
            $dayEnd = (clone $dayStart)->modify('+1 day');
            if ($dayEnd > $endDate) {
                $dayEnd = clone $endDate;
            }

            // Calculate seconds for this slice
            $intervalStart = $current > $dayStart ? $current : $dayStart;
            $intervalSeconds = $dayEnd->getTimestamp() - $intervalStart->getTimestamp();

            if ($dayOfWeek >= 6) { // Saturday or Sunday
                $weekendSeconds += $intervalSeconds;
            }

            // Move to next day start
            $current = $dayEnd;
        }

        $businessSeconds = $totalSeconds - $weekendSeconds;

        // Convert to hours with decimals
        $businessHours = max(0, $businessSeconds / 3600);

        return $businessHours;
    }

    private function calculateTotalHoursDiff($start, $end)
    {
        $startDate = new DateTime($start);
        $endDate = new DateTime($end);

        if ($endDate < $startDate) {
            return 0;
        }

        $totalSeconds = $endDate->getTimestamp() - $startDate->getTimestamp();
        return $totalSeconds / 3600; // convert to hours
    }


    private function paginateArray(array $items, $perPage = 25, $page = null)
    {
        $page = $page ?: (LengthAwarePaginator::resolveCurrentPage() ?: 1);
        $itemsCollection = collect($items);
        $paginatedItems = $itemsCollection->slice(($page - 1) * $perPage, $perPage)
            ->map(function ($item) {
                return (object) $item; // convert array to object
            })
            ->values();

        return new LengthAwarePaginator(
            $paginatedItems,
            count($items),
            $perPage,
            $page,
            ['path' => LengthAwarePaginator::resolveCurrentPath()]
        );
    }

    private function calculateAppraiserStats($assignments)
    {
        $appraisers = [];

        foreach ($assignments as $assignment) {
            $userId = $assignment->user_id;

            if (!isset($appraisers[$userId])) {
                $appraisers[$userId] = [
                    'user_id' => $userId,
                    'first_name' => $assignment->name,
                    'last_name' => '',
                    'total_assignments' => 0,
                    'total_completes' => 0,
                    'total_open' => 0,
                    'total_completion_hours' => 0,
                    'completed_count' => 0,
                ];
            }

            $appraisers[$userId]['total_assignments']++;

            if (in_array($assignment->status_id, [$this->completed_status_id, $this->closed_status_id])) {
                $appraisers[$userId]['total_completes']++;

                if($this->exclude_weekend){
                    $hours = $this->calculateBusinessHoursDiff($assignment->created_date, $assignment->complete_date);
                }else{
                    $hours = $this->calculateTotalHoursDiff($assignment->created_date, $assignment->complete_date);
                }



                $appraisers[$userId]['total_completion_hours'] += $hours;
                $appraisers[$userId]['completed_count']++;
            } elseif (!in_array($assignment->status_id, [$this->completed_status_id, $this->closed_status_id, $this->cancelled_status_id])) {
                $appraisers[$userId]['total_open']++;
            }
        }

        $appraisers = array_filter($appraisers, function ($appraiser) {
            return $appraiser['completed_count'] > 0;
        });
        
        // Calculate average completion hours
        foreach ($appraisers as &$appraiser) {
            $appraiser['avg_completion_hours'] = $appraiser['completed_count'] > 0
                ? $appraiser['total_completion_hours'] / $appraiser['completed_count']
                : null;
        }


        unset($appraiser); // break reference

        // Sort by avg_completion_hours ascending, nulls last
        usort($appraisers, function ($a, $b) {
            if ($a['avg_completion_hours'] === null) return 1;
            if ($b['avg_completion_hours'] === null) return -1;
            return $a['avg_completion_hours'] <=> $b['avg_completion_hours'];
        });

        // Return paginated results
        return $this->paginateArray($appraisers, 25);
    }

    private function calculateAppraiserStatsRaw($assignments)
    {
        $appraisers = [];

        foreach ($assignments as $assignment) {
            $userId = $assignment->user_id;

            if (!isset($appraisers[$userId])) {
                $appraisers[$userId] = [
                    'user_id' => $userId,
                    'first_name' => $assignment->name,
                    'last_name' => '',
                    'total_assignments' => 0,
                    'total_completes' => 0,
                    'total_open' => 0,
                    'total_completion_hours' => 0,
                    'completed_count' => 0,
                ];
            }

            $appraisers[$userId]['total_assignments']++;

            if (in_array($assignment->status_id, [$this->completed_status_id, $this->closed_status_id])) {
                $appraisers[$userId]['total_completes']++;

                if($this->exclude_weekend){
                    $hours = $this->calculateBusinessHoursDiff($assignment->created_date, $assignment->complete_date);
                }else{
                    $hours = $this->calculateTotalHoursDiff($assignment->created_date, $assignment->complete_date);
                }

                $appraisers[$userId]['total_completion_hours'] += $hours;
                $appraisers[$userId]['completed_count']++;
            } elseif (!in_array($assignment->status_id, [$this->completed_status_id, $this->closed_status_id, $this->cancelled_status_id])) {
                $appraisers[$userId]['total_open']++;
            }
        }

        $appraisers = array_filter($appraisers, function ($appraiser) {
            return $appraiser['completed_count'] > 0;
        });
        
        // Calculate average completion hours
        foreach ($appraisers as &$appraiser) {
            $appraiser['avg_completion_hours'] = $appraiser['completed_count'] > 0
                ? $appraiser['total_completion_hours'] / $appraiser['completed_count']
                : null;
        }

        unset($appraiser); // break reference

        // Sort by avg_completion_hours ascending, nulls last
        usort($appraisers, function ($a, $b) {
            if ($a['avg_completion_hours'] === null) return 1;
            if ($b['avg_completion_hours'] === null) return -1;
            return $a['avg_completion_hours'] <=> $b['avg_completion_hours'];
        });

        // Return raw array without pagination
        return $appraisers;
    }

    private function calculateAverages($appraiserData)
    {
        if (empty($appraiserData)) {
            return [
                'avg_total_assignments' => 0,
                'avg_total_open' => 0,
                'avg_total_completes' => 0,
                'avg_cycle_time' => 0
            ];
        }

        $totalAssignments = 0;
        $totalOpen = 0;
        $totalCompletes = 0;
        $totalCycleTime = 0;
        $validCycleTimeCount = 0;

        foreach ($appraiserData as $appraiser) {
            $totalAssignments += ($appraiser['total_open'] + $appraiser['total_completes']);
            $totalOpen += $appraiser['total_open'];
            $totalCompletes += $appraiser['total_completes'];
            
            if ($appraiser['avg_completion_hours'] !== null) {
                $totalCycleTime += $appraiser['avg_completion_hours'];
                $validCycleTimeCount++;
            }
        }

        $count = count($appraiserData);

        return [
            'avg_total_assignments' => $count > 0 ? round($totalAssignments / $count, 2) : 0,
            'avg_total_open' => $count > 0 ? round($totalOpen / $count, 2) : 0,
            'avg_total_completes' => $count > 0 ? round($totalCompletes / $count, 2) : 0,
            'avg_cycle_time' => $validCycleTimeCount > 0 ? round($totalCycleTime / $validCycleTimeCount, 2) : 0
        ];
    }

    private function calculateTotals($appraiserData)
    {
        if (empty($appraiserData)) {
            return [
                'total_assignments' => 0,
                'total_open' => 0,
                'total_completes' => 0,
                'total_cycle_time' => 0
            ];
        }

        $totalAssignments = 0;
        $totalOpen = 0;
        $totalCompletes = 0;
        $totalCycleTime = 0;

        foreach ($appraiserData as $appraiser) {
            $totalAssignments += ($appraiser['total_open'] + $appraiser['total_completes']);
            $totalOpen += $appraiser['total_open'];
            $totalCompletes += $appraiser['total_completes'];
            
            if ($appraiser['avg_completion_hours'] !== null) {
                $totalCycleTime += $appraiser['avg_completion_hours'];
            }
        }

        return [
            'total_assignments' => $totalAssignments,
            'total_open' => $totalOpen,
            'total_completes' => $totalCompletes,
            'total_cycle_time' => $totalCycleTime
        ];
    }

    private function getFranchises()
    {
        if (count($this->multiple_franchise) > 0) {
            return DB::table('franchise')->select('id', 'franchisecode', 'company_name')->orderBy('franchisecode', 'asc')->whereIn('id', $this->multiple_franchise)->get();
        } else {
            return DB::table('franchise')->select('id', 'franchisecode', 'company_name')->orderBy('franchisecode', 'asc')->get();
        }
    }

    private function getInsuranceCompanies()
    {
        return DB::table('insurance_company')->orderBy('company_name', 'asc')->get();
    }
}

